
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)


# Define the vector field function for population density flow
def population_vector_field(X, Y):
    # Example function for population flow
    U = np.cos(2 * np.pi * X) * np.sin(2 * np.pi * Y)
    V = np.sin(2 * np.pi * X) * np.cos(2 * np.pi * Y)
    return U, V


def modified_population_vector_field(X, Y):
    # Modified population flow with some external effect
    U = np.abs(np.cos(2 * np.pi * X) * np.sin(2 * np.pi * Y))
    V = np.abs(np.sin(2 * np.pi * X) * np.cos(2 * np.pi * Y))
    return U, V


# Create a grid of points representing regions
x = np.linspace(0, 1, 10)
y = np.linspace(0, 1, 10)
X, Y = np.meshgrid(x, y)

# Compute the vector fields
U, V = population_vector_field(X, Y)
U_mod, V_mod = modified_population_vector_field(X, Y)

# Plot curves representing different population growth paths
x = np.linspace(0.3, 0.6, 100)
xlabel = "Distance (km)"
ylabel = "Population Density (people/km²)"
patch_labels = ["Normal Population Flow", "Modified Population Flow"]
line_labels = ["Growth Path 1", "Growth Path 2", "Decline Path 1", "Decline Path 2"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(10, 8))

# Quiver plots for population density flow
plt.quiver(X, Y, U, V, color="#1f77b4", alpha=0.8)  # Normal Flow (blue color)
plt.quiver(
    X, Y, U_mod, V_mod, color="#2ca02c", alpha=0.8
)  # Modified Flow (green color)

# Plot different population growth paths
plt.plot(x, 0.09 / (x**1.2), color="#ff7f0e", linestyle="-", linewidth=2)
plt.plot(x, 0.08 / (x**1.2 + 0.04), color="#d62728", linestyle="--", linewidth=2)
plt.plot(x, 0.075 / (x**1 + 0.04), color="#9467bd", linestyle="-.", linewidth=2)
plt.plot(x, 0.12 / (x**1 + 0.05), color="#8c564b", linestyle=":", linewidth=2)

# Add labels and legend
plt.xlabel(xlabel, fontsize=16, style="italic")
plt.ylabel(ylabel, fontsize=16, style="italic")

# Patches for vector fields
blue_patch = mpatches.Patch(color="#1f77b4", label=patch_labels[0], alpha=0.8)
green_patch = mpatches.Patch(color="#2ca02c", label=patch_labels[1], alpha=0.8)

# Lines for population growth paths
main_path_1 = mlines.Line2D(
    [], [], color="#ff7f0e", linestyle="-", linewidth=2, label=line_labels[0]
)
main_path_2 = mlines.Line2D(
    [], [], color="#d62728", linestyle="--", linewidth=2, label=line_labels[1]
)
alt_path_1 = mlines.Line2D(
    [], [], color="#9467bd", linestyle="-.", linewidth=2, label=line_labels[2]
)
alt_path_2 = mlines.Line2D(
    [], [], color="#8c564b", linestyle=":", linewidth=2, label=line_labels[3]
)

# Combine all legend handles
handles = [blue_patch, green_patch, main_path_1, main_path_2, alt_path_1, alt_path_2]

# Add the legend to the plot with specified location
plt.legend(handles=handles, loc="upper right")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("quiver_53.pdf", bbox_inches="tight")
