
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Data for Revenue, Profit, and Expenses over 10 months
months = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct"]
revenue = np.random.uniform(20000, 30000, 10).tolist()
profit = np.random.uniform(5000, 15000, 10).tolist()
expenses = np.random.uniform(10000, 20000, 10).tolist()

num_vars = len(months)
labels = ["Revenue ($)", "Profit ($)", "Expenses ($)"]
ticks = [5000, 10000, 15000, 20000, 25000, 30000]
tickslabel = ["5k", "10k", "15k", "20k", "25k", "30k"]

title = "Sales Data Over 10 Months"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Compute angle for each axis
angles = np.linspace(0, 2 * np.pi, num_vars, endpoint=False).tolist()

# The plot is circular, so we need to "complete the loop" and append the start to the end.
revenue += revenue[:1]
profit += profit[:1]
expenses += expenses[:1]
angles += angles[:1]

# Draw the radar chart
fig, ax = plt.subplots(figsize=(8, 8), subplot_kw=dict(polar=True))

# Colors
revenue_color = "royalblue"
profit_color = "seagreen"
expenses_color = "darkorange"

# Revenue
ax.fill(angles, revenue, color=revenue_color, alpha=0.25)
ax.plot(
    angles,
    revenue,
    color=revenue_color,
    linewidth=2,
    linestyle="-",
    marker="o",
    label=labels[0],
)
ax.scatter(angles[:-1], revenue[:-1], color=revenue_color, s=50)

# Profit
ax.fill(angles, profit, color=profit_color, alpha=0.25)
ax.plot(
    angles,
    profit,
    color=profit_color,
    linewidth=2,
    linestyle="--",
    marker="s",
    label=labels[1],
)
ax.scatter(angles[:-1], profit[:-1], color=profit_color, s=50)

# Expenses
ax.fill(angles, expenses, color=expenses_color, alpha=0.25)
ax.plot(
    angles,
    expenses,
    color=expenses_color,
    linewidth=2,
    linestyle="-.",
    marker="^",
    label=labels[2],
)
ax.scatter(angles[:-1], expenses[:-1], color=expenses_color, s=50)

# Add labels to the plot
ax.set_yticklabels([])
ax.set_xticks(angles[:-1])
ax.set_xticklabels(months)

# Add grid lines and labels for the concentric circles
ax.set_rgrids(
    ticks,
    labels=tickslabel,
    angle=30,
    color="gray",
    size=10,
)

# Create legend handles manually
legend_elements = [
    Line2D(
        [0],
        [0],
        color=revenue_color,
        linewidth=2,
        marker="o",
        linestyle="-",
        markersize=8,
        label=labels[0],
    ),
    Line2D(
        [0],
        [0],
        color=profit_color,
        linewidth=2,
        marker="s",
        linestyle="--",
        markersize=8,
        label=labels[1],
    ),
    Line2D(
        [0],
        [0],
        color=expenses_color,
        linewidth=2,
        marker="^",
        linestyle="-.",
        markersize=8,
        label=labels[2],
    ),
]

# Add legend and title
ax.legend(
    handles=legend_elements, loc="upper right", bbox_to_anchor=(1.1, 1.1), frameon=False
)
ax.set_title(title, size=20, color="darkblue")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("radar_29.pdf", bbox_inches="tight")
