
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)  # Set a random seed for reproducibility

# Generate mean number of tourists (in thousands) for five cities
mean_tourists = [200, 150, 300, 250, 180]
# Use suitable standard deviations to reflect the distribution of tourists
standard_deviations = [20, 15, 25, 30, 18]

tourist_numbers1 = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_tourists, standard_deviations)
]
tourist_numbers2 = [
    np.random.normal(loc=mean + 10, scale=std, size=50)  # Assume a slight increase in the second year
    for mean, std in zip(mean_tourists, standard_deviations)
]
positions1 = np.array(range(1, len(tourist_numbers1) + 1)) - 0.2
positions2 = np.array(range(1, len(tourist_numbers2) + 1)) + 0.2
legend_labels = ["Year 2022", "Year 2023"]
xlabel = "Cities"
ylabel = "Number of Tourists (in thousands)"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["New York", "Paris", "Tokyo", "London", "Sydney"]


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with the specified dimensions
fig, ax = plt.subplots(figsize=(8, 7))

# Narrower violin plots
violin_width = 0.35

# Create the violin plot with adjusted positions
parts1 = ax.violinplot(
    tourist_numbers1, positions=positions1, widths=violin_width, showmeans=False, showmedians=True
)
parts2 = ax.violinplot(
    tourist_numbers2, positions=positions2, widths=violin_width, showmeans=False, showmedians=True
)

# Customizing the colors of the violin plot
for pc in parts1["bodies"]:
    pc.set_facecolor("#a1dab4")  # Light Green
    pc.set_edgecolor("black")
    pc.set_alpha(1)

for pc in parts2["bodies"]:
    pc.set_facecolor("#41b6c4")  # Light Blue
    pc.set_edgecolor("black")
    pc.set_alpha(1)

# Customizing the median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = parts1[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

    vp = parts2[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

# Customizing the colors of the violin plot and adding statistics
for i in range(len(tourist_numbers1)):
    # Adding the statistical annotations for tourist_numbers1
    quartile1, median, quartile3 = np.percentile(tourist_numbers1[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(tourist_numbers1[i][tourist_numbers1[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(tourist_numbers1[i][tourist_numbers1[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions1[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(
        median,
        positions1[i] - 0.025,
        positions1[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions1[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1
    )

    # Adding the statistical annotations for tourist_numbers2
    quartile1, median, quartile3 = np.percentile(tourist_numbers2[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(tourist_numbers2[i][tourist_numbers2[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(tourist_numbers2[i][tourist_numbers2[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions2[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(
        median,
        positions2[i] - 0.025,
        positions2[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions2[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1
    )

# Change the border color to grey
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove small ticks beside the numbers on the x and y axes
ax.tick_params(axis="both", which="both", length=0)

# Adding the legend
ax.legend(
    [parts1["bodies"][0], parts2["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting the title and labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting the x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size
fig.set_size_inches(7, 5)  # Width, Height in inches

# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_21.pdf", bbox_inches="tight")
