
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# New data simulating temperature changes in different cities
temperatures_before_intervention = np.random.normal(loc=[15, 20, 10, 25, 30], scale=[2, 3, 1.5, 2.5, 3], size=(100, 5))
temperatures_after_intervention = np.random.normal(loc=[14, 19, 9, 24, 29], scale=[2, 3, 1.5, 2.5, 3], size=(100, 5))

# Y-axis label and limits
ylabel = "Temperature (°C)"
ylim = [0, 40]
violin_width = 0.5
scaling_factor = 1
kde_x = np.linspace(0, 40, 300)

# Offsets for groups
offsets = np.linspace(-3, 3, 5)
labels = ["Before Intervention", "After Intervention"]
titles = ["Temperature Changes by City"]
legend_labels = ["After Intervention", "Before Intervention"]
xticklabels = ["New York", "Los Angeles", "Chicago", "Houston", "Phoenix"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure
fig, ax = plt.subplots(1, 1, figsize=(10, 6))

# Define the colors for the two distributions
colors = ["#1f77b4", "#ff7f0e"]

# Function to plot half violins
def plot_half_violin(ax, data, data_memory, offsets, colors, labels, title, xticklabels):
    for i in range(data.shape[1]):
        offset = offsets[i]

        # Plot data before intervention
        kde_data = gaussian_kde(data[:, i])
        kde_x = np.linspace(0, 40, 300)
        kde_data_y = kde_data(kde_x)
        kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            kde_data_y_scaled * scaling_factor + offset,
            offset,
            color=colors[0],
            edgecolor='#2a9df4',
            alpha=0.7
        )
        
        # Plot data after intervention
        kde_data_memory = gaussian_kde(data_memory[:, i])
        kde_data_memory_y = kde_data_memory(kde_x)
        kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
        ax.fill_betweenx(
            kde_x,
            offset,
            -kde_data_memory_y_scaled * scaling_factor + offset,
            color=colors[1],
            edgecolor='#ff9731',
            alpha=0.7
        )

    # Set x and y axis labels, limits, and add x-axis tick labels for groups
    ax.set_xlim(
        min(offsets) - scaling_factor - violin_width,
        max(offsets) + scaling_factor + violin_width,
    )
    ax.set_ylim(ylim)
    ax.set_ylabel(ylabel)
    ax.set_xticks(offsets)
    ax.set_xticklabels(xticklabels)
    ax.title.set_text(title)

# Plot the violins
plot_half_violin(ax, temperatures_before_intervention, temperatures_after_intervention, offsets, colors, labels, titles[0], xticklabels)

# Add a legend for the entire figure
handles = [
    plt.Line2D([0], [0], marker='o', color='w', markerfacecolor=colors[1], markersize=10),
    plt.Line2D([0], [0], marker='o', color='w', markerfacecolor=colors[0], markersize=10),
]

fig.legend(handles, legend_labels, loc='lower center', bbox_to_anchor=(0.5, -0.1), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Tighten the layout and show the combined plot
plt.tight_layout()
plt.savefig("violin_27.pdf", bbox_inches="tight")
