
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
import random

np.random.seed(42)

# Generate data for temperature variations across cities
data = np.random.normal(loc=[15, 20, 25, 30, 35], scale=[5, 4, 3, 6, 5], size=(10, 5))
data_memory = np.random.normal(
    loc=[17, 22, 27, 32, 37], scale=[6, 5, 4, 7, 6], size=(40, 5)
)

# X-axis labels and legend labels
xticklabels = ["City A", "City B", "City C", "City D", "City E"]
legend_labels = ["Without Heating", "With Heating"]

# The scaling factor and violin width
scaling_factor = 1
violin_width = 0.5

# Adjust the offsets for 5 groups
offsets = np.linspace(-3, 3, 5)

# Chart labels and title
xlabel = "Cities"
ylabel = "Temperature (°C)"
title = "Temperature Variation in Different Cities"
supertitle = "Temperature Data: Without Heating vs. With Heating"
legend_title = "Condition"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(8, 6))  # Adjusted figure size

# Define new colors for the artistic theme
colors = ["#6a0dad", "#ff4500"]
legend_colors = ["#ff4500", "#6a0dad"]

# Plot the half-violins with an offset for 5 groups
for i in range(data.shape[1]):
    offset = offsets[i]

    # Plot data without memory (Without Heating)
    kde_data = gaussian_kde(data[:, i])
    kde_x = np.linspace(-10, 50, 300)
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="black",
        alpha=0.7,
    )

    # Plot data with memory (With Heating)
    kde_data_memory = gaussian_kde(data_memory[:, i])
    kde_data_memory_y = kde_data_memory(kde_x)
    kde_data_memory_y_scaled = kde_data_memory_y / max(kde_data_memory_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_memory_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="black",
        alpha=0.7,
    )

    # Add yellow stars at the top of each violin plot
    ax.scatter(
        offset,
        random.uniform(-5, 45),
        marker="*",
        color="yellow",
        s=260,
        zorder=3,
        edgecolor="black",
    )

# Set x and y axis labels, limits, and add x-axis tick labels for 5 groups
ax.set_xlim(
    min(offsets) - scaling_factor - violin_width,
    max(offsets) + scaling_factor + violin_width,
)
ax.set_xticks(offsets)  # Set x-ticks to the center of each group
ax.set_xticklabels(xticklabels)  # Label x-ticks accordingly
ax.set_xlabel(xlabel)  # Set x-axis label
ax.set_ylabel(ylabel)  # Set y-axis label

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black")
    for color in legend_colors
]

ax.legend(handles, legend_labels, loc="upper left", ncol=1, title=legend_title)

# Set the title and supertitle
plt.title(title)
plt.suptitle(supertitle)

# Enhance grid for better readability
ax.grid(True, linestyle="--", alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("violin_33.pdf", bbox_inches="tight")
