
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Sample data to mimic case duration and settlement amount
duration_data = np.random.beta(
    a=[3, 5, 7, 6, 4], b=[10, 15, 20, 25, 12], size=(10, 5)
)
settlement_data = np.random.beta(a=[5, 8, 10, 6, 7], b=[20, 25, 15, 18, 20], size=(40, 5))
xticklabels = ["Criminal", "Civil", "Family", "Corporate", "International"]
legend_labels = ["Case Duration", "Settlement Amount"]
# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1
violin_width = 0.5

# Adjust the offsets for 5 groups instead of 3
offsets = np.linspace(-3, 3, 5)

# Text labels
xlabel = "Case Type"
ylabel = "Normalized Density"
title = "Distribution of Case Duration and Settlement Amounts"
legend_title = "Metrics"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(8, 6))

# Define the colors for each group
colors = ["#7fbf7b", "#af8dc3"]
legend_colors = ["#af8dc3", "#7fbf7b"]

# Plot the half-violins with an offset for 5 groups
for i in range(duration_data.shape[1]):
    offset = offsets[i]

    # Plot duration data
    kde_duration = gaussian_kde(duration_data[:, i])
    kde_x = np.linspace(0, 1, 300)
    kde_duration_y = kde_duration(kde_x)
    kde_duration_y_scaled = kde_duration_y / max(kde_duration_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_duration_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="black",
        alpha=0.6,
    )

    # Plot settlement data
    kde_settlement = gaussian_kde(settlement_data[:, i])
    kde_settlement_y = kde_settlement(kde_x)
    kde_settlement_y_scaled = kde_settlement_y / max(kde_settlement_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_settlement_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="black",
        alpha=0.6,
    )

    # Add circle markers at the top of each violin plot
    ax.scatter(
        offset,
        np.random.uniform(0.2, 0.8),
        marker="o",
        color="red",
        s=80,
        zorder=3,
        edgecolor="black",
    )

# Set x and y axis labels, limits, and add x-axis tick labels for 5 groups
ax.set_xlim(
    min(offsets) - scaling_factor - violin_width,
    max(offsets) + scaling_factor + violin_width,
)
ax.set_xticks(offsets)  # Set x-ticks to the center of each group
ax.set_xticklabels(xticklabels)  # Label x-ticks accordingly
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black", alpha=0.6)
    for color in legend_colors
]
ax.legend(handles, legend_labels, title=legend_title, loc="upper left", ncol=1)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig("violin_35.pdf", bbox_inches="tight")
