
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Sample software performance data for demonstration purposes
scenarios = np.arange(1, 6)
version_a_data = (
    np.random.randn(10, 100) * 5 + 50
)  # Centered around 50 (e.g., ms for response time) with some noise
version_b_data = (
    np.random.randn(10, 100) * 6 + 45
)  # Centered around 45 (e.g., ms for response time) with some noise

titles = [
    "Response Time vs. Version A",
    "Response Time vs. Version B",
]
ylims = [[30, 70], [30, 70]]
xlabel = "Scenarios"
# ytickslabels = [
#     ["30ms", "40ms", "50ms", "60ms", "70ms"],
#     ["30ms", "40ms", "50ms", "60ms", "70ms"],
# ]
xticklabel = ["Scenario 1", "Scenario 2", "Scenario 3", "Scenario 4", "Scenario 5"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and subplots
fig, axs = plt.subplots(2, 1, figsize=(10, 10))

# Define the colors for the violin plots
color_version_a = "#228B22"  # ForestGreen for Version A
color_version_b = "#4682B4"  # SteelBlue for Version B


# Function to set the color of the violin plot
def set_violin_color(violin, color):
    for body in violin["bodies"]:
        body.set_facecolor(color)
        body.set_edgecolor(color)
    # Set color for the median line
    violin["cmedians"].set_color(color)
    violin["cmedians"].set_linewidth(2)


# Top subplot: Version A
for i, scenario in enumerate(scenarios):
    vl = axs[0].violinplot(
        version_a_data[i],
        positions=[scenario - 0.2],
        showmedians=True,
        widths=0.3,
        showextrema=False,
    )
    set_violin_color(vl, color_version_a)

    vl = axs[0].violinplot(
        version_b_data[i],
        positions=[scenario + 0.2],
        showmedians=True,
        widths=0.3,
        showextrema=False,
    )
    set_violin_color(vl, color_version_b)

    axs[0].text(
        scenario - 0.35,
        np.median(version_a_data[i]),
        f"{int(np.median(version_a_data[i]))}ms",
        ha="right",
        va="bottom",
        color=color_version_a,
    )
    axs[0].text(
        scenario + 0.35,
        np.median(version_b_data[i]),
        f"{int(np.median(version_b_data[i]))}ms",
        ha="left",
        va="bottom",
        color=color_version_b,
    )

axs[0].set_title(titles[0])
axs[0].set_xticks(scenarios)
axs[0].set_ylim(ylims[0])
axs[0].set_xticklabels([])
# axs[0].set_yticklabels(ytickslabels[0])

# Bottom subplot: Version B
for i, scenario in enumerate(scenarios):
    vl = axs[1].violinplot(
        version_a_data[i],
        positions=[scenario - 0.2],
        showmedians=True,
        widths=0.3,
        showextrema=False,
    )
    set_violin_color(vl, color_version_a)

    vl = axs[1].violinplot(
        version_b_data[i],
        positions=[scenario + 0.2],
        showmedians=True,
        widths=0.3,
        showextrema=False,
    )
    set_violin_color(vl, color_version_b)

    axs[1].text(
        scenario - 0.35,
        np.median(version_a_data[i]),
        f"{int(np.median(version_a_data[i]))}ms",
        ha="right",
        va="bottom",
        color=color_version_a,
    )
    axs[1].text(
        scenario + 0.35,
        np.median(version_b_data[i]),
        f"{int(np.median(version_b_data[i]))}ms",
        ha="left",
        va="bottom",
        color=color_version_b,
    )

axs[1].set_title(titles[1])
axs[1].set_xticks(scenarios)
axs[1].set_ylim(ylims[1])
axs[1].set_xlabel(xlabel)
axs[1].set_xticklabels(xticklabel)
# axs[1].set_yticklabels(ytickslabels[1])

# use line to create a custom legend
legend_elements = [
    Line2D([0], [0], color=color_version_a, lw=2, label="Version A"),
    Line2D([0], [0], color=color_version_b, lw=2, label="Version B"),
]
axs[0].legend(handles=legend_elements, loc="upper right")
axs[1].legend(handles=legend_elements, loc="upper right")

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save plot
plt.tight_layout()
plt.savefig("violin_40.pdf", bbox_inches="tight")
