
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)
# Generate new sample data for survey of daily electricity consumption across different types of households
household_group1 = np.clip(np.random.normal(15, 5, 200), 0, 30)  # Small Apartments
household_group2 = np.clip(np.random.normal(25, 7, 200), 0, 40)  # Medium Houses
household_group3 = np.clip(np.random.normal(35, 8, 200), 0, 50)  # Large Houses
household_group4 = np.clip(np.random.normal(45, 10, 200), 0, 60)  # Villas
household_group5 = np.clip(np.random.normal(30, 6, 200), 0, 50)  # Townhouses

correlation_factors = [0.35, 0.45, 0.40, 0.38, 0.42]
energy_efficiency_rating = [7.2, 7.8, 7.5, 7.6, 7.4]

data = [household_group1, household_group2, household_group3, household_group4, household_group5]
categories = ["Small Apt", "Medium House", "Large House", "Villa", "Townhouse"]
ylabel = "Daily Electricity Consumption (kWh)"
ylim = [0, 70]
xlabel = "Household Types"
textlabels = ["Corr. Factor", "Energy Eff. Rating(%)"]
title = "Survey of Daily Electricity Consumption Across Household Types"
supertitle = "Electricity Consumption Survey"
legendlabels = ["Corr. Factor", "Energy Eff. Rating(%)"]
text_color_corr = "darkorange"
text_color_eff_rating = "darkblue"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))  # Adjust the figure size to accommodate more violins

# Create violin plots
violin_parts = ax.violinplot(data, showmeans=False, showmedians=True, showextrema=False)

# Customize the appearance
ax.set_ylabel(ylabel)
ax.set_xticks(np.arange(1, len(categories) + 1))  # Adjust the x-ticks to match the number of categories
ax.set_xticklabels(categories)
ax.set_ylim(ylim)  # You may need to adjust this if the data range changes
ax.set_xlabel(xlabel)
ax.set_title(title)
fig.suptitle(supertitle)

# Set violin colors and add statistical annotations
colors = ["#1f77b4", "#ff7f0e", "#2ca02c", "#d62728", "#9467bd"]  # New color scheme

for i, (pc, d) in enumerate(zip(violin_parts['bodies'], data)):
    pc.set_facecolor(colors[i])
    pc.set_edgecolor('black')
    pc.set_alpha(1)

    # Calculate the quartiles and median
    quartile1, median, quartile3 = np.percentile(d, [25, 50, 75])
    iqr = quartile3 - quartile1

    # Calculate whiskers
    lower_whisker = np.min(d[d >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(d[d <= quartile3 + 1.5 * iqr])

    # Annotate statistics
    ax.vlines(i + 1, quartile1, quartile3, color='k', linestyle='-', lw=4)
    ax.scatter(i + 1, median, color='w', s=10, zorder=3)
    ax.vlines(i + 1, lower_whisker, upper_whisker, color='k', linestyle='-', lw=1)
    ax.text(i + 1 + 0.3, np.median(data[i]), f"{median:.2f}", ha="left", va="center", color="black", rotation=45)

    # Annotate with correlation factors and energy efficiency rating values
    ax.text(i + 1, 65, f"{correlation_factors[i]:.2f}", ha="center", va="center", color=text_color_corr, fontsize=10)
    ax.text(i + 1, 3, f"{energy_efficiency_rating[i]:.2f}", ha="center", va="center", color=text_color_eff_rating, fontsize=10)

ax.text(5.6, 65, textlabels[0], ha="left", va="center", color=text_color_corr, fontsize=10)
ax.text(5.6, 3, textlabels[1], ha="left", va="center", color=text_color_eff_rating, fontsize=10)

# Make the other parts of the violin plots invisible
for partname in ('cbars', 'cmins', 'cmaxes', 'cmedians'):
    vp = violin_parts.get(partname)
    if vp:
        vp.set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig("violin_43.pdf", bbox_inches="tight")
