
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)
# Generate sample data for agricultural metrics
data = np.random.normal(loc=[3.5, 7.5, 2.8], scale=[0.8, 1.2, 0.5], size=(100, 3))
data_treated = np.random.normal(loc=[4.0, 8.0, 3.2], scale=[0.7, 1.1, 0.6], size=(100, 3))
kde_x = np.linspace(0, 12, 300)

# Define the categories and the colors for each group
categories = ["Wheat Yield", "Corn Yield", "Soybean Yield"]
title = "Agricultural Yield Analysis: Traditional vs. Organic Farming"
offsets = [-0.05, 0, 0.05]
ylabel = "Yield (tons per hectare)"
labels = ["Traditional Farming", "Organic Farming"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(figsize=(8, 6))
colors = ["skyblue", "lightgreen"]

# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1
violin_width = 0.015

# Plot the half-violins with an offset
for i, category in enumerate(categories):
    offset = offsets[i]
    # Plot data without treatment
    kde_data = gaussian_kde(data[:, i])
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="black",
    )

    # Plot data with treatment
    kde_data_treated = gaussian_kde(data_treated[:, i])
    kde_data_treated_y = kde_data_treated(kde_x)
    kde_data_treated_y_scaled = kde_data_treated_y / max(kde_data_treated_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_treated_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="black",
    )

    # Plot the mean as a star marker for data without treatment
    ax.plot(
        offset,
        np.mean(data[:, i]),
        "*",
        color="white",
        markersize=12,
        markeredgecolor="black",
    )
    # Plot the mean as a star marker for data with treatment
    ax.plot(
        offset,
        np.mean(data_treated[:, i]),
        "*",
        color="white",
        markersize=12,
        markeredgecolor="black",
    )

    ax.text(
        offset, -0.1, category, ha="center", va="top", fontsize=9
    )  # Add the category label below the violin plot

# Add title
ax.set_title(title, fontsize=14)

# Add grid
ax.grid(True, linestyle="--", alpha=0.7)

# Set x and y axis labels and limits
ax.set_xlim(
    min(offsets) - scaling_factor * violin_width - 0.06,
    max(offsets) + scaling_factor * violin_width + 0.06,
)
ax.set_ylim(0, 12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_xticks([])  # Remove x-ticks as they are not meaningful here

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="black") for color in colors
]

ax.legend(handles, labels, loc="lower center", bbox_to_anchor=(0.5, -0.15), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit and save the figure
plt.tight_layout()
plt.savefig("violin_53.pdf", bbox_inches="tight")
