

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate mean values for time spent on activities (in minutes per day)
mean_values_teenagers = np.linspace(30, 120, 5)
mean_values_adults = np.linspace(40, 90, 5)

# Standard deviations
standard_deviations = [15] * 5

# Generate synthetic data for Teenagers
data_teenagers = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_teenagers, standard_deviations)
]

# Generate synthetic data for Adults
data_adults = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_adults, standard_deviations)
]

# Set positions for violins
positions_teenagers = np.array(range(1, len(data_teenagers) + 1)) - 0.2
positions_adults = np.array(range(1, len(data_adults) + 1)) + 0.2

# Labels and ticks
legend_labels = ["Teenagers", "Adults"]
xlabel = "Entertainment Activities"
ylabel = "Time Spent (Minutes per Day)"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Watching TV", "Playing Video Games", "Reading", "Social Media", "Listening to Music"]
title = "Time Spent on Entertainment Activities by Age Group"
suitxt = "Entertainment Activity Analysis"
# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(8, 7))

violin_width = 0.35

# Create violin plots
parts_teenagers = ax.violinplot(
    data_teenagers,
    positions=positions_teenagers,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)
parts_adults = ax.violinplot(
    data_adults,
    positions=positions_adults,
    widths=violin_width,
    showmeans=False,
    showmedians=True,
)

# Customizing colors
for pc in parts_teenagers["bodies"]:
    pc.set_facecolor("teal")
    pc.set_edgecolor("black")
    pc.set_alpha(1)

for pc in parts_adults["bodies"]:
    pc.set_facecolor("orange")
    pc.set_edgecolor("black")
    pc.set_alpha(1)

# Customizing median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp_teenagers = parts_teenagers[partname]
    vp_adults = parts_adults[partname]
    vp_teenagers.set_edgecolor("black")
    vp_adults.set_edgecolor("black")
    vp_teenagers.set_linewidth(1)
    vp_adults.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp_teenagers.set_visible(False)  # Hide caps
        vp_adults.set_visible(False)  # Hide caps

# Adding statistical annotations
for i in range(len(data_teenagers)):
    quartile1, median, quartile3 = np.percentile(data_teenagers[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_teenagers[i][data_teenagers[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_teenagers[i][data_teenagers[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_teenagers[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_teenagers[i] - 0.025,
        positions_teenagers[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_teenagers[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

    quartile1, median, quartile3 = np.percentile(data_adults[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_adults[i][data_adults[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_adults[i][data_adults[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(
        positions_adults[i], quartile1, quartile3, color="black", linestyle="-", lw=4
    )
    ax.hlines(
        median,
        positions_adults[i] - 0.025,
        positions_adults[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_adults[i],
        lower_whisker,
        upper_whisker,
        color="black",
        linestyle="-",
        lw=1,
    )

# Customize borders
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove ticks
ax.tick_params(axis="both", which="both", length=0)

# Adding legend
ax.legend(
    [parts_teenagers["bodies"][0], parts_adults["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.0),
    ncol=2,
)

# Setting labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(True, linestyle="-", linewidth=0.7, color="grey", zorder=0)

# Adding title
plt.title(title)
plt.suptitle(suitxt, y=0.98, fontsize=13)

# ===================
# Part 4: Saving Output
# ===================
fig.set_size_inches(7, 5)

plt.tight_layout()
plt.savefig("violin_58.pdf", bbox_inches="tight")
