
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(24)
# Generate synthetic data for religion domain
participation_rates = np.random.beta(a=2, b=5, size=100)
donation_rates = np.random.beta(a=5, b=2, size=100)
community_service_rates = np.random.beta(a=3, b=3, size=100)
data = np.vstack([participation_rates, donation_rates, community_service_rates]).T

participation_rates_optimized = np.random.beta(a=5, b=2, size=100)
donation_rates_optimized = np.random.beta(a=2, b=5, size=100)
community_service_rates_optimized = np.random.beta(a=3, b=3, size=100)
data_optimized = np.vstack(
    [participation_rates_optimized, donation_rates_optimized, community_service_rates_optimized]
).T

categories = ["Participation Rates", "Donation Rates", "Community Service Rates"]
violin_width = 0.015

# Axes Limits and Labels
ylabel_value = "Engagement Score"
labels = ["Traditional Approach", "Modern Approach"]
title = "Engagement Comparison Between Religious Approaches"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
fig, ax = plt.subplots(
    figsize=(6, 6)
)  # Use the subplots function to create a figure and single axes

# Define the categories and the colors for each group
colors = ["#8dbd8b", "#c7a76e"]

# The scaling factor is used to ensure the violins do not overlap
scaling_factor = 1

# Define offset to separate the half violin plots in the single Axes object
offsets = [-0.05, 0, 0.05]

# Plot the half-violins with an offset
for i, category in enumerate(categories):
    offset = offsets[i]

    # Plot traditional method data
    kde_data = gaussian_kde(data[:, i])
    kde_x = np.linspace(0, 1, 300)
    kde_data_y = kde_data(kde_x)
    kde_data_y_scaled = kde_data_y / max(kde_data_y) * violin_width
    ax.fill_betweenx(
        kde_x,
        kde_data_y_scaled * scaling_factor + offset,
        offset,
        color=colors[0],
        edgecolor="#6a948a",
    )

    # Plot optimized method data
    kde_data_optimized = gaussian_kde(data_optimized[:, i])
    kde_data_optimized_y = kde_data_optimized(kde_x)
    kde_data_optimized_y_scaled = (
        kde_data_optimized_y / max(kde_data_optimized_y) * violin_width
    )
    ax.fill_betweenx(
        kde_x,
        offset,
        -kde_data_optimized_y_scaled * scaling_factor + offset,
        color=colors[1],
        edgecolor="#b3a77a",
    )
    ax.text(
        offset, -0.1, category, ha="center", va="top"
    )  # Add the category label below the violin plot

# Set x and y axis labels and limits
ax.set_xlim(
    min(offsets) - scaling_factor * violin_width - 0.01,
    max(offsets) + scaling_factor * violin_width + 0.01,
)
y_margin = 0.01  # Adding 1% margin at top and bottom of the y-axis
y_min, y_max = ax.get_ylim()
ax.set_ylim(y_min - y_margin, y_max + y_margin)
ax.set_ylabel(ylabel_value)
ax.set_xticks([])  # Remove x-ticks as they are not meaningful here

# Set a title for the chart
ax.set_title(title)

# Adjust the legend
handles = [
    plt.Rectangle((0, 0), 1, 1, color=color, edgecolor="#9e8d8b") for color in colors
]
ax.legend(handles, labels, loc="lower left", bbox_to_anchor=(0.6, -0.2), ncol=1)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout for better fit and save the plot
plt.tight_layout()
plt.savefig("violin_64.pdf", bbox_inches="tight")
