

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(79)
# Simulate new data representing tourism metrics
tourism_data1 = np.clip(np.random.normal(0.70, 0.10, 200), 0, 1)  # Hotels
tourism_data2 = np.clip(np.random.normal(0.65, 0.12, 200), 0, 1)  # Airlines
tourism_data3 = np.clip(np.random.normal(0.75, 0.09, 200), 0, 1)  # Car Rentals
tourism_data4 = np.clip(np.random.normal(0.68, 0.11, 200), 0, 1)  # Travel Agencies
tourism_data5 = np.clip(np.random.normal(0.72, 0.08, 200), 0, 1)  # Tour Operators

# Simulated metrics for Customer Satisfaction and Engagement Rate
customer_satisfaction = [0.76, 0.68, 0.80, 0.72, 0.74]
engagement_rate = [60, 55, 70, 65, 68]

data = [tourism_data1, tourism_data2, tourism_data3, tourism_data4, tourism_data5]
categories = ["Hotels", "Airlines", "Car Rentals", "Travel Agencies", "Tour Operators"]
ylabel = "Tourism Metrics"
ylim = [0, 1.06]
xlabel = "Tourism Segment"
textlabels = ["Customer Satisfaction", "Engagement Rate (%)"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(12, 7))

# Create violin plots
violin_parts = ax.violinplot(data, showmeans=False, showmedians=True, showextrema=False)

# Customize the appearance
ax.set_ylabel(ylabel)
ax.set_xticks(np.arange(1, len(categories) + 1))
ax.set_xticklabels(categories)
ax.set_ylim(ylim)
ax.set_xlabel(xlabel)

# Define a vibrant entertainment-oriented color palette
colors = ["#e41a1c", "#377eb8", "#4daf4a", "#984ea3", "#ff7f00"]

for i, (pc, d) in enumerate(zip(violin_parts["bodies"], data)):
    pc.set_facecolor(colors[i])
    pc.set_edgecolor("black")
    pc.set_alpha(0.75)
    
    # Calculate the quartiles and median
    quartile1, median, quartile3 = np.percentile(d, [25, 50, 75])
    iqr = quartile3 - quartile1

    # Calculate whiskers
    lower_whisker = np.min(d[d >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(d[d <= quartile3 + 1.5 * iqr])

    # Annotate statistics
    ax.vlines(i + 1, quartile1, quartile3, color="k", linestyle="-", lw=4)
    ax.scatter(i + 1, median, color="w", s=40, zorder=3)
    ax.vlines(i + 1, lower_whisker, upper_whisker, color="k", linestyle="-", lw=1)
    ax.text(i + 1 + 0.3, np.median(data[i]), f"{median:.2f}", ha="left", va="center", color="black", rotation=45)

    # Annotate with Customer Satisfaction and Engagement Rate values
    ax.text(i + 1, 0.14, f"{customer_satisfaction[i]:.2f}", ha="center", va="center", color="green", fontsize=10)
    ax.text(i + 1, 0.08, f"{engagement_rate[i]:.2f}", ha="center", va="center", color="blue", fontsize=10)

ax.text(5.6, 0.14, textlabels[0], ha="left", va="center", color="green", fontsize=10)
ax.text(5.6, 0.08, textlabels[1], ha="left", va="center", color="blue", fontsize=10)

# Make the other parts of the violin plots invisible
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = violin_parts.get(partname)
    if vp:
        vp.set_visible(False)

# Add grid for better readability
ax.grid(True, linestyle="--", which="both", color="grey", alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("violin_79.pdf", bbox_inches="tight")

