
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Simulate new data representing success rates of various medical treatments
health_data1 = np.clip(np.random.normal(0.70, 0.06, 200), 0, 1)  # Drug A
health_data2 = np.clip(np.random.normal(0.35, 0.05, 200), 0, 1)  # Drug B
health_data3 = np.clip(np.random.normal(0.50, 0.06, 200), 0, 1)  # Surgery A
health_data4 = np.clip(np.random.normal(0.75, 0.07, 200), 0, 1)  # Surgery B
health_data5 = np.clip(np.random.normal(0.50, 0.08, 200), 0, 1)  # Therapy

# Simulated metrics for Pearson R and Error Rate (EER)
pearson_r = [0.45, 0.42, 0.40, 0.35, 0.30]
eer = [1.5, 2.0, 2.5, 3.0, 3.5]

data = [
    health_data1,
    health_data2,
    health_data3,
    health_data4,
    health_data5,
]
categories = ["Drug A", "Drug B", "Surgery A", "Surgery B", "Therapy"]
ylabel = "Success Rate"
ylim = [0, 1.06]
xlabel = "Treatment Type"
title = "Comparative Analysis of Medical Treatments' Success Rates"
textlabels = ["Pearson Correlation (R)", "Error Rate (EER) (%)"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 6))

# Create violin plots
violin_parts = ax.violinplot(data, showmeans=False, showmedians=True, showextrema=False)

# Customize the appearance
ax.set_ylabel(ylabel)
ax.set_xticks(np.arange(1, len(categories) + 1))
ax.set_xticklabels(categories)
ax.set_ylim(ylim)
ax.set_xlabel(xlabel)
ax.set_title(title)

# Define a health-oriented color palette
colors = ["#66c2a5", "#fc8d62", "#8da0cb", "#e78ac3", "#a6d854"]

for i, (pc, d) in enumerate(zip(violin_parts["bodies"], data)):
    pc.set_facecolor(colors[i])
    pc.set_edgecolor("black")
    pc.set_alpha(0.75)

    # Calculate the quartiles and median
    quartile1, median, quartile3 = np.percentile(d, [25, 50, 75])
    iqr = quartile3 - quartile1

    # Calculate whiskers
    lower_whisker = np.min(d[d >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(d[d <= quartile3 + 1.5 * iqr])

    # Annotate statistics
    ax.vlines(i + 1, quartile1, quartile3, color="k", linestyle="-", lw=4)
    ax.scatter(i + 1, median, color="w", s=40, zorder=3)
    ax.vlines(i + 1, lower_whisker, upper_whisker, color="k", linestyle="-", lw=1)
    ax.text(
        i + 1 + 0.3,
        np.median(data[i]),
        f"{median:.2f}",
        ha="left",
        va="center",
        color="black",
        rotation=45,
    )

    # Annotate with Pearson R and EER values
    ax.text(
        i + 1,
        0.14,
        f"{pearson_r[i]:.2f}",
        ha="center",
        va="center",
        color="green",
        fontsize=10,
    )
    ax.text(
        i + 1,
        0.08,
        f"{eer[i]:.2f}",
        ha="center",
        va="center",
        color="blue",
        fontsize=10,
    )

ax.text(5.6, 0.14, textlabels[0], ha="left", va="center", color="green", fontsize=10)
ax.text(5.6, 0.08, textlabels[1], ha="left", va="center", color="blue", fontsize=10)

# Make the other parts of the violin plots invisible
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = violin_parts.get(partname)
    if vp:
        vp.set_visible(False)

# Add grid for better readability
ax.grid(True, linestyle="--", which="both", color="grey", alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("violin_80.pdf", bbox_inches="tight")
