import matplotlib.pyplot as plt
import numpy as np; np.random.seed(0); np.random.seed(0)

# ===================
# Part 2: Data Preparation
# ===================
# Simulate business data for revenue and profit margins
# Assuming a more realistic scenario with different parameters
k = 0.4
h = 10  # Base revenue value
a = 5  # Adjusted to fit a business revenue-profits relationship
profit_margins = np.linspace(5, 20, 400)  # Profit margins from 5% to 20%
revenue = a * (profit_margins - k) ** 2 + h  # Revenue function based on profit margins

# Increasing the number of points for density
points_profit = np.linspace(7, 18, 20)  # Denser profit margin points for higher resolution
points_revenue = a * (points_profit - k) ** 2 + h - 1.5  # Offset to simulate variability

# Labels and Plot Types
label_1 = "Revenue vs. Profit Margin"
label_2 = "± 3-σ of Prediction"
label_3 = "Data Points"
ylim_values = [5, 20]
ellipse_sizes = np.linspace(15, 15, len(points_profit))  # Varied sizes of ellipses

# Axes Limits and Labels
xlabel_value = "Revenue (in millions)"
ylabel_value = "Profit Margin (%)"
title = "Revenue vs. Profit Margin with Uncertainty Ellipses"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Adjusting the size of the error ellipses to make them larger and more clearly elliptical, with black borders
plt.figure(figsize=(8, 6))
plt.plot(revenue, profit_margins, label=label_1, color="red")

# Making error ellipses perfectly circular
ellipse_sizes = ellipse_sizes

# Adding error ellipses
for i, (px, py, size) in enumerate(zip(points_revenue, points_profit, ellipse_sizes)):
    if i == 0:  # Only for the first ellipse add label
        ellipse = plt.matplotlib.patches.Ellipse(
            (px, py),
            width=size,
            height=size / 4,
            edgecolor="black",
            facecolor="none",
            label=label_2,
        )
    else:
        ellipse = plt.matplotlib.patches.Ellipse(
            (px, py), width=size, height=size / 4, edgecolor="black", facecolor="none"
        )
    plt.gca().add_patch(ellipse)

plt.scatter(points_revenue, points_profit, color="blue", label=label_3, zorder=5)
plt.xlabel(xlabel_value)
plt.ylabel(ylabel_value)
plt.title(title)
plt.ylim(ylim_values)
plt.grid(True, color="white")
plt.gca().set_facecolor("#eaeaf2")
plt.gca().set_axisbelow(True)

# Adjusting the legend to include the representative ellipse
plt.legend(facecolor="#eaeaf2")
for spine in plt.gca().spines.values():
    spine.set_visible(False)
plt.gca().tick_params(axis="both", length=0)  # Hide tick marks

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig('HR_4.pdf', bbox_inches='tight')
