# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)
# Sample data for 24 hours, with half-hour steps
time_step = np.linspace(0, 24, 48)
glucose_levels = 90 + 10 * np.sin(np.pi * time_step / 12) + np.random.normal(0, 5, 48)
insulin_levels = 15 + 5 * np.cos(np.pi * time_step / 12) + time_step * 0.2

# Extracted variables for labels and settings
glucose_label = "Glucose Levels (mg/dL)"
insulin_label = "Insulin Levels (mU/L)"
glucose_ylabel = "Glucose Levels (mg/dL)"
insulin_xlabel = "Time (hours)"
insulin_ylabel = "Insulin Levels (mU/L)"
legend_labels = ["Glucose Levels", "Insulin Levels"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axes
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(10, 10))

glucose_ylim = [70, 120]
glucose_xlim = [0, 24]
glucose_yticks = [70, 80, 90, 100, 110, 120]
glucose_xticks = np.arange(0, 25, 4)

insulin_ylim = [0, 30]
insulin_xlim = [0, 24]
insulin_yticks = [0, 5, 10, 15, 20, 25, 30]
insulin_xticks = np.arange(0, 25, 4)

legend_location = "lower center"
legend_bbox_to_anchor = (0.5, -0.2)
legend_ncol = 2
legend_frameon = False
marker_size = 6
alpha_fill = 0.3
grid_status = True
tick_length = 0
bbox_inches = "tight"

# Plot Glucose Levels
ax1.plot(time_step, glucose_levels, "o-", color="#d62728", label=glucose_label)
ax1.fill_between(time_step, glucose_levels, color="#ff9896", alpha=alpha_fill)
ax1.set_ylim(glucose_ylim)
ax1.set_xlim(glucose_xlim)
ax1.set_yticks(glucose_yticks)
ax1.set_xticks(glucose_xticks)
ax1.set_ylabel(glucose_ylabel)
ax1.tick_params(axis="both", which="both", length=tick_length)
ax1.grid(grid_status)

# Plot Insulin Levels
ax2.plot(time_step, insulin_levels, "s--", color="#2ca02c", label=insulin_label)
ax2.fill_between(time_step, insulin_levels, color="#98df8a", alpha=alpha_fill)
ax2.set_ylim(insulin_ylim)
ax2.set_xlim(insulin_xlim)
ax2.set_yticks(insulin_yticks)
ax2.set_xticks(insulin_xticks)
ax2.set_xlabel(insulin_xlabel)
ax2.set_ylabel(insulin_ylabel)
ax2.tick_params(axis="both", which="both", length=tick_length)
ax2.grid(grid_status)

# Create custom legend
red_line = mlines.Line2D(
    [], [], color="#d62728", marker="o", markersize=marker_size, label=legend_labels[0]
)
green_line = mlines.Line2D(
    [], [], color="#2ca02c", marker="s", markersize=marker_size, label=legend_labels[1]
)
plt.legend(
    handles=[red_line, green_line],
    loc=legend_location,
    bbox_to_anchor=legend_bbox_to_anchor,
    ncol=legend_ncol,
    frameon=legend_frameon,
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the figure
plt.tight_layout()
plt.savefig("area_20.pdf", bbox_inches=bbox_inches)