# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
# Data configuration
import numpy as np
np.random.seed(42)

# Grid
x = np.linspace(-15, 15, 300)
y = np.linspace(-15, 15, 300)
X, Y = np.meshgrid(x, y)

# Transmitter locations
transmitters = [(-10, -10), (0, 10), (10, -10), (15, 15), (-15, 5)]

# Signal strength function
def signal_strength(x, y, transmitters):
    Z = np.zeros_like(x)
    for tx, ty in transmitters:
        Z += np.exp(-((x - tx)**2 + (y - ty)**2) / 30)
    return Z

# Calculate Z as signal strength
Z = signal_strength(X, Y, transmitters)

# Titles and labels
plot_title = "Radio Signal Intensity"
xlabel = "Longitude"
ylabel = "Latitude"
legend_location = "lower left"
colorbar_label = "Signal Strength"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Contour
cnt = ax.contour(X, Y, Z, cmap="plasma", linewidths=1)
contour_filled = ax.contourf(X, Y, Z, cmap="plasma", alpha=0.7)
ax.clabel(cnt, cnt.levels, inline=True, fontsize=10, fmt="%.1f")

# Transmitter locations
for tx, ty in transmitters:
    ax.plot(tx, ty, 'bo', markersize=12, label=f'Transmitter ({tx},{ty})')

# Adding color bar
cbar = fig.colorbar(contour_filled, ax=ax)
cbar.set_label(colorbar_label)

# Titles and labels
ax.set_title(plot_title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.legend(loc=legend_location)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("contour_17.pdf", bbox_inches="tight")