# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)
# Data for plotting
companies = [
    "QuantumTech Inc.",
    "Renewable Future Co.",
    "BioGen Labs",
    "CyberNet Global",
    "NanoSys Solutions",
    "HydroFlow Energy",
    "AeroFusion Dynamics",
    "EcoStream Systems",
    "FusionPower Ltd.",
    "GeoThermal Ventures",
]
energy_outputs = [
    350.5,  # Gigawatt hours (GWh)
    420.1,
    390.8,
    455.5,
    320.3,
    405.9,
    440.3,
    375.7,
    415.0,
    365.6,
]
error_margins = [
    [15.0, -12.8],  # Error margins in GWh
    [18.5, -16.2],
    [12.9, -10.7],
    [17.4, -14.1],
    [11.8, -9.6],
    [13.0, -11.8],
    [16.3, -13.0],
    [14.0, -10.7],
    [15.2, -12.9],
    [13.0, -11.8],
]
forecast_methods = [
    "Energy Consumption Estimates",
    "Sustainability Projections",
    "Renewable Integration Trends",
    "AI-Based Models",
    "Technology Forecasts",
    "Market Adjustments",
    "Hybrid Simulations",
    "Algorithmic Predictions",
    "Green Energy Analysis",
    "Long-Term Forecast",
]
xticks = np.arange(300.0, 500.0, 20.0)  # Adjusted for energy scale
xlim = [300.0, 500.0]
industry_avg_output = 350.0  # Industry average energy output in GWh
highlight_range = [340.0, 360.0]  # Range near industry average
xlabel = "Population Growth Rate (%)"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plotting
fig, ax = plt.subplots(figsize=(10, 8))

# Error bars with different positive and negative values
for i, (company, energy_output, error_margin) in enumerate(zip(companies, energy_outputs, error_margins)):
    ax.errorbar(
        energy_output,
        i,
        xerr=[[abs(error_margin[1])], [error_margin[0]]],
        fmt="o",
        color="black",
        ecolor="black",
        capsize=3,
    )
    ax.text(
        energy_output,
        i - 0.15,
        r"$%.1f^{+%.2f} _{-%.2f}$" % (energy_output, error_margin[0], abs(error_margin[1])),
        va="center",
        ha="center",
        fontsize=9,
    )

# Highlighted region with adjusted color and alpha
ax.axvspan(highlight_range[0], highlight_range[1], color="yellow", alpha=0.3)

# Text for methods with adjusted font size
for i, method in enumerate(forecast_methods):
    ax.text(510, i, method, va="center", ha="left", fontsize=11)

# Set labels and title
ax.set_yticks(range(len(companies)))
ax.set_yticklabels(companies)
ax.set_xlabel(xlabel, fontsize=12)
ax.set_xlim(xlim)
ax.invert_yaxis()  # Invert y-axis to match the original image
ax.axvline(x=industry_avg_output, linestyle="--", color="orange")
# Adjust x-axis ticks and labels
ax.set_xticks(xticks)
ax.set_xticklabels([f"{x:.1f}" for x in xticks])

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save/show the plot
plt.tight_layout()
plt.savefig("errorpoint_13.pdf", bbox_inches="tight")