# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import networkx as nx

# Create a directed graph for department workflow in a tech company
department_graph = nx.DiGraph()

# Add nodes with their respective colors and labels for different departments
departments = {
    0: ("Development", "lightcoral"),
    1: ("Testing", "dodgerblue"),
    2: ("Deployment", "lightgreen"),
    3: ("Support", "gold"),
    4: ("HR", "violet"),
    5: ("Finance", "orange"),
    6: ("Legal", "darkred"),
}
for dept_id, (dept_name, dept_color) in departments.items():
    department_graph.add_node(dept_id, label=dept_name, color=dept_color)

# Add edges with labels representing the workflow between departments
workflow_edges = [
    (0, 1, "to Testing"), 
    (1, 2, "to Deployment"), 
    (2, 3, "to Support"),
    (3, 4, "to HR"),
    (4, 5, "to Finance"),
    (5, 6, "to Legal")
]
for src, dest, flow_label in workflow_edges:
    department_graph.add_edge(src, dest, label=flow_label)

# Define node positions in a circular layout
positions = nx.circular_layout(department_graph)

# Title for the plot
plot_title = "Department Workflow in a Tech Company"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw nodes with the color attribute
department_colors = [department_graph.nodes[dept]["color"] for dept in department_graph.nodes]
nx.draw_networkx_nodes(department_graph, positions, node_color=department_colors, node_size=7000, ax=ax)

# Draw edges with arrows indicating direction of workflow
nx.draw_networkx_edges(department_graph, positions, arrowstyle="->", arrowsize=20, ax=ax)

# Draw department labels
department_labels = {dept: department_graph.nodes[dept]["label"] for dept in department_graph.nodes}
nx.draw_networkx_labels(department_graph, positions, labels=department_labels, font_size=12, font_color="white", font_weight="bold", ax=ax)

# Draw edges with workflow labels
workflow_labels = {(src, dest): department_graph[src][dest]["label"] for src, dest in department_graph.edges}
nx.draw_networkx_edge_labels(department_graph, positions, edge_labels=workflow_labels, font_size=10, ax=ax)

# Add a title to the plot
plt.title(plot_title, size=15, fontweight="bold")

# Remove axis for a cleaner look
plt.axis("off")

# ===================
# Part 4: Saving Output
# ===================
# Display the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_7.pdf", bbox_inches="tight")