# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
# Create a directed graph
import networkx as nx

network_graph = nx.DiGraph()

# Add nodes with their respective colors representing network components
network_components = {
    0: {"color": "lightcoral", "type": "Router"},
    1: {"color": "lightseagreen", "type": "Server"},
    2: {"color": "gold", "type": "Switch"},
    3: {"color": "orchid", "type": "Firewall"},
    4: {"color": "deepskyblue", "type": "Database"},
    5: {"color": "darkorange", "type": "Client"},
    6: {"color": "forestgreen", "type": "Load Balancer"},
}
for component_id, attributes in network_components.items():
    network_graph.add_node(component_id, color=attributes["color"], type=attributes["type"])

# Add edges with labels representing types of network communication
communication_edges = [
    (0, 1, "Data Transfer"), 
    (1, 2, "Latency"), 
    (2, 3, "Packet Filtering"), 
    (3, 4, "Data Query"), 
    (4, 5, "Response Time"), 
    (5, 6, "Load Balancing")
]
for source, target, communication_type in communication_edges:
    network_graph.add_edge(source, target, label=communication_type)

# Define node positions in a circular layout
layout_positions = nx.circular_layout(network_graph)

new_title = "Network Components and Communication Flow"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(10, 10))

# Draw nodes with color attribute
component_colors = [network_graph.nodes[component_id]["color"] for component_id in network_graph.nodes]
nx.draw_networkx_nodes(network_graph, layout_positions, node_color=component_colors, node_size=500)

# Draw edges with labels
nx.draw_networkx_edges(network_graph, layout_positions, arrows=True, connectionstyle='arc3,rad=0.2')
edge_labels = {(source, target): network_graph[source][target]["label"] for source, target in network_graph.edges}
nx.draw_networkx_edge_labels(network_graph, layout_positions, edge_labels=edge_labels)

# Add node labels
nx.draw_networkx_labels(network_graph, layout_positions, {component_id: component_id for component_id in network_graph.nodes})

# Add a title
plt.title(new_title)

# Create a legend for node types
legend_handles = [
    plt.Line2D([0], [0], marker='o', color='w', markerfacecolor=color, label=type, markersize=10)
    for color, type in set((attributes["color"], attributes["type"]) for attributes in network_components.values())
]
ax.legend(handles=legend_handles, loc='upper right')

# Remove axis
plt.axis("off")

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("graph_8.pdf", bbox_inches="tight")