import numpy as np; np.random.seed(0); np.random.seed(0)

import matplotlib.pyplot as plt
from matplotlib.colors import LogNorm

# ===================
# Part 2: Data Preparation
# ===================
# Defining the colormap from white to blue
cmap = plt.cm.Blues

# Data for the two subplots
data1 = np.array(
    [
        [8.3, 12.0, 25.9, 7.0, 18.4, 28.3],
        [4.2, 11.2, 30.6, 9.8, 21.3, 22.9],
        [6.8, 15.3, 20.9, 12.2, 25.5, 19.3],
        [10.0, 12.1, 15.5, 14.1, 30.1, 18.2],
        [15.7, 18.1, 10.6, 24.5, 20.6, 10.5],
    ]
)

data2 = np.array(
    [
        [11.5, 7.0, 18.7, 10.9, 25.0, 26.9],
        [9.4, 10.0, 17.8, 19.9, 22.0, 21.9],
        [8.6, 12.2, 16.8, 21.3, 24.1, 17.0],
        [7.3, 14.2, 14.0, 20.5, 26.0, 18.8],
        [6.5, 13.3, 20.4, 22.5, 23.1, 14.3],
    ]
)

# X and Y labels
x_labels = ["Diabetes", "Heart Disease", "Cancer", "Stroke", "Respiratory Disease", "Other"]
y_labels = ["Diabetes", "Heart Disease", "Cancer", "Stroke", "Respiratory Disease"]
# Subplot titles
titles = ["(a) Health condition distribution by age group", "(b) Final health state of age groups"]

# Set up the colormap and norm (log scale)
norm = LogNorm(vmin=0.1, vmax=100)

# Axes Limits and Labels
xticks_values = range(len(x_labels))
yticks_values = range(len(y_labels))
colorbar_ticks = [0.1, 1, 10, 100]
yticklabels = ["0.1", "1", "10", "100"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set up the figure
fig, axes = plt.subplots(
    1, 2, figsize=(20, 8), gridspec_kw={"width_ratios": [1, 1], "wspace": 0.3}
)


# Function to create a subplot
def create_subplot(ax, data, title):
    # Create the scatter plot
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            # Calculate the color based on the original value
            color = cmap(norm(data[i, j]))

            # Draw the circle with a fixed size
            circle = plt.Circle((j, i), 0.4, color=color)  # Fixed size
            ax.add_artist(circle)

            # Determine text color based on the value
            text_color = "white" if data[i, j] > 25 else "black"

            # Add the text inside the circle
            ax.text(
                j, i, f"{data[i, j]:.1f}%", ha="center", va="center", color=text_color
            )

    # Set labels for x and y axes
    ax.set_xticks(range(len(x_labels)))
    ax.set_xticklabels(x_labels, ha="center")
    ax.set_yticks(range(len(y_labels)))
    ax.set_yticklabels(y_labels, va="center")

    # Adding the title for the subplot
    ax.set_title(title, fontsize=16)

    # Set the limits of the axes; they should be one more than your data range
    ax.set_xlim(-0.5, data.shape[1] - 0.5)
    ax.set_ylim(-0.5, data.shape[0] - 0.5)

    # Set the aspect of the plot to be equal and add a frame
    ax.set_aspect("equal")
    for spine in ax.spines.values():
        spine.set_visible(True)


# Create each subplot
create_subplot(axes[0], data1, titles[0])
create_subplot(axes[1], data2, titles[1])

# Create a colorbar on the far right side of the figure
sm = plt.cm.ScalarMappable(cmap=cmap, norm=norm)
sm.set_array([])
cbar = fig.colorbar(
    sm,
    ax=axes,
    ticks=colorbar_ticks,
    orientation="vertical",
    fraction=0.015,
    pad=0.05,
)
cbar.ax.set_yticklabels(yticklabels)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig('heatmap_19.pdf', bbox_inches='tight')
