import matplotlib.pyplot as plt
import numpy as np; np.random.seed(0); np.random.seed(0)

import matplotlib
import matplotlib as mpl

# ===================
# Part 2: Data Preparation
# ===================
cities = ["City A", "City B", "City C", "City D", "City E", "City F", "City G"]
months = [
    "January",
    "February",
    "March",
    "April",
    "May",
    "June",
    "July",
]

temperature = np.array(
    [
        [5.2, 6.1, 7.3, 9.2, 12.8, 15.4, 18.2],
        [3.1, 4.5, 7.8, 11.3, 14.7, 17.8, 19.6],
        [2.2, 5.0, 9.1, 12.2, 16.0, 20.5, 22.1],
        [6.3, 8.1, 10.2, 13.5, 17.4, 22.0, 24.8],
        [1.9, 3.8, 6.9, 10.6, 15.3, 18.9, 21.7],
        [0.5, 2.3, 5.2, 9.1, 13.6, 17.2, 20.9],
        [3.3, 5.7, 8.6, 12.8, 16.5, 20.0, 23.4],
    ]
)

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

def heatmap(
    data, row_labels, col_labels, ax=None, cbar_kw=None, cbarlabel="", **kwargs
):
    if ax is None:
        ax = plt.gca()

    if cbar_kw is None:
        cbar_kw = {}

    # Plot the heatmap
    im = ax.imshow(data, **kwargs)

    # Create colorbar
    cbar = ax.figure.colorbar(im, ax=ax, **cbar_kw)
    cbar.ax.set_ylabel(cbarlabel, rotation=-90, va="bottom")

    # Show all ticks and label them with the respective list entries.
    ax.set_xticks(np.arange(data.shape[1]), labels=col_labels)
    ax.set_yticks(np.arange(data.shape[0]), labels=row_labels)

    # Let the horizontal axes labeling appear on top.
    ax.tick_params(top=True, bottom=False, labeltop=True, labelbottom=False)

    # Rotate the tick labels and set their alignment.
    plt.setp(ax.get_xticklabels(), rotation=-30, ha="right", rotation_mode="anchor")

    # Turn spines off and create white grid.
    ax.spines[:].set_visible(False)

    ax.set_xticks(np.arange(data.shape[1] + 1) - 0.5, minor=True)
    ax.set_yticks(np.arange(data.shape[0] + 1) - 0.5, minor=True)
    ax.grid(which="minor", color="w", linestyle="-", linewidth=3)
    ax.tick_params(which="minor", bottom=False, left=False)

    return im, cbar


def annotate_heatmap(
    im,
    data=None,
    valfmt="{x:.1f}",
    textcolors=("black", "white"),
    threshold=None,
    **textkw
):
    if not isinstance(data, (list, np.ndarray)):
        data = im.get_array()

    # Normalize the threshold to the images color range.
    if threshold is not None:
        threshold = im.norm(threshold)
    else:
        threshold = im.norm(data.max()) / 2.0

    # Set default alignment to center, but allow it to be
    # overwritten by textkw.
    kw = dict(horizontalalignment="center", verticalalignment="center")
    kw.update(textkw)

    # Get the formatter in case a string is supplied
    if isinstance(valfmt, str):
        valfmt = matplotlib.ticker.StrMethodFormatter(valfmt)

    # Loop over the data and create a `Text` for each "pixel".
    # Change the text's color depending on the data.
    texts = []
    for i in range(data.shape[0]):
        for j in range(data.shape[1]):
            kw.update(color=textcolors[int(im.norm(data[i, j]) > threshold)])
            text = im.axes.text(j, i, valfmt(data[i, j], None), **kw)
            texts.append(text)

    return texts


fig, ax = plt.subplots(figsize=(8, 8))
im, cbar = heatmap(
    temperature, cities, months, ax=ax, cmap="YlGn", cbarlabel="Average Temperature [°C]"
)
texts = annotate_heatmap(im, valfmt="{x:.1f} °C")

# ===================
# Part 4: Saving Output
# ===================
fig.tight_layout()
plt.savefig('heatmap_24.pdf', bbox_inches='tight')
