import matplotlib.pyplot as plt
from matplotlib.lines import Line2D  # Importing Line2D for creating custom legend items

# ===================
# Part 2: Data Preparation
# ===================
# Data
test_drives = [2, 3, 4, 5, 6, 7, 8]
sedan_efficiency = [30, 32, 33, 34, 35, 36, None]  # None value will be handled in the plot commands
suv_efficiency = [20, 22, 23, 24, 25, 26, 27]
truck_efficiency = [15] * len(test_drives)
ev_efficiency = [25] * len(test_drives)

# Axes Limits and Labels
xlabel_value = "Number of Test Drives"
ylabel_value_1 = "Sedan Fuel Efficiency (MPG)"
ylabel_value_2 = "SUV Fuel Efficiency (MPG)"
ylim_values_1 = [10, 40]
ylim_values_2 = [10, 30]
yticks_values_1 = range(15, 41, 5)
yticks_values_2 = range(15, 31, 5)

# Labels
label_1 = "Sedan"
label_2 = "SUV"
label_3 = "Truck"
label_4 = "Electric Vehicle"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Plot
fig, ax1 = plt.subplots(figsize=(8, 7))

# Sedan plot
(sedan_line,) = ax1.plot(
    test_drives,
    sedan_efficiency,
    "o-",
    color="#377eb8",
    label=label_1,
    markersize=10,
    linewidth=3,
    mec="black",
)
ax1.set_xlabel(xlabel_value, fontsize=14)
ax1.set_ylabel(ylabel_value_1, fontsize=14, color="#377eb8")
ax1.tick_params(
    axis="y", labelcolor="#377eb8", direction="in", rotation=90, labelsize=12
)
ax1.tick_params(
    axis="x",
    direction="in",
    labelsize=12,
)
ax1.set_yticks(yticks_values_1)
ax1.set_ylim(ylim_values_1)

# Adding fuel efficiency values to the plot for sedan
for i, txt in enumerate(sedan_efficiency):
    if txt is not None:  # Skip plotting the text for None values
        ax1.annotate(
            f"{txt} MPG",
            (test_drives[i], txt),
            textcoords="offset points",
            xytext=(10, 10),
            ha="center",
            fontsize=12,
        )

# SUV plot with a secondary y-axis
ax2 = ax1.twinx()
(suv_line,) = ax2.plot(
    test_drives,
    suv_efficiency,
    "^-",
    color="#ff7f00",
    label=label_2,
    markersize=10,
    linewidth=3,
    mec="black",
)
ax2.set_ylabel(ylabel_value_2, color="#ff7f00", fontsize=14)
ax2.tick_params(
    axis="y", labelcolor="#ff7f00", direction="in", rotation=90, labelsize=12
)
ax2.set_yticks(yticks_values_2)
ax2.set_ylim(ylim_values_2)

# Adding fuel efficiency values to the plot for SUV
for i, txt in enumerate(suv_efficiency):
    ax2.annotate(
        f"{txt} MPG",
        (test_drives[i], txt),
        textcoords="offset points",
        xytext=(0, -30),
        ha="center",
        fontsize=12,
    )

# Truck dashed lines
ax1.axhline(y=15, color="#377eb8", linestyle=":", linewidth=2)
ax1.axhline(y=15.5, color="#ff7f00", linestyle=":", linewidth=2)

# EV dashed lines
ax1.axhline(y=25, color="#377eb8", linestyle="--", linewidth=2)
ax1.axhline(y=26, color="#ff7f00", linestyle="--", linewidth=2)

# Creating custom legend items
truck_legend = Line2D([0], [0], color="black", linestyle=":", linewidth=2, label="Truck")
ev_legend = Line2D([0], [0], color="black", linestyle="--", linewidth=2, label="Electric Vehicle")

# Adding legends
first_legend = ax1.legend(
    handles=[truck_legend, ev_legend],
    loc="upper left",
    ncol=2,
    fontsize=14,
    edgecolor="black",
)
ax1.add_artist(first_legend)  # Add the first legend manually
second_legend = ax1.legend(
    handles=[sedan_line, suv_line], loc="upper right", fontsize=14, edgecolor="black"
)  # Add the second legend

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig('line_34.pdf', bbox_inches='tight')
