import matplotlib.pyplot as plt
import numpy as np; np.random.seed(0); np.random.seed(0)


# ===================
# Part 2: Data Preparation
# ===================
# Updated steps
time_steps = np.linspace(0, 24, 50)  # Time of day in hours

# Generating distinct trends for each line
traffic_flow = np.sin(time_steps * 0.5) * 100 + 500  # Sinusoidal traffic flow pattern
fuel_consumption = np.array(time_steps) ** 2 * 0.5 + 200  # Quadratic fuel consumption
accident_rate = np.array([153.53, 150.8, 151.96, 154.48, 153.74, 148.05, 151.9, 149.7, 149.79, 150.82, 150.29, 152.91, 151.52, 150.24, 150.89, 150.67, 152.99, 149.59, 150.63, 148.29, 144.89, 151.31, 151.73, 148.52, 154.54, 147.09, 150.09, 149.63, 153.07, 152.94, 150.31, 150.76, 148.22, 146.04, 149.3, 150.31, 152.46, 152.4, 149.23, 149.4, 147.9, 147.16, 146.59, 153.9, 148.98, 149.12, 147.49, 151.55, 146.77, 149.57])  # Random noise for accidents
passenger_count = np.exp(0.05 * time_steps) * 20  # Exponential growth in passenger count

# Simulating standard deviations for error
std_dev = 20
traffic_flow_std = np.full_like(traffic_flow, std_dev)
fuel_consumption_std = np.full_like(fuel_consumption, std_dev)
accident_rate_std = np.full_like(accident_rate, std_dev)
passenger_count_std = np.full_like(passenger_count, std_dev)

# Axes Limits and Labels
xlabel_value = "Time of Day (hours)"
xlim_values = [0, 24]
xticks_values = np.linspace(0, 24, 9)

ylabel_value_1 = "Traffic Flow (vehicles/hour)"
ylabel_value_2 = "Passenger Count (thousands)"
yticks_values_1 = np.arange(0, 1100, 200)
yticks_values_2 = np.arange(0, 400, 40)
ylim_values_1 = [0, 1000]
ylim_values_2 = [0, 400]

# Labels
label_1 = "Traffic Flow (Sinusoidal)"
label_2 = "Fuel Consumption (Quadratic)"
label_3 = "Accident Rate (Random Noise)"


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating a figure with two subplots
fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(6, 8))

# Plotting on the first subplot
ax1.plot(
    time_steps, traffic_flow, "o-", color="purple", label=label_1
)
ax1.fill_between(
    time_steps,
    traffic_flow - traffic_flow_std,
    traffic_flow + traffic_flow_std,
    color="purple",
    alpha=0.2,
)

ax1.plot(time_steps, fuel_consumption, "s-", color="blue", label=label_2)
ax1.fill_between(
    time_steps,
    fuel_consumption - fuel_consumption_std,
    fuel_consumption + fuel_consumption_std,
    color="blue",
    alpha=0.2,
)

ax2.plot(
    time_steps,
    accident_rate,
    "^--",
    color="green",
    markerfacecolor=(0, 0, 0, 0),
    markeredgecolor="green",
    label=label_3,
)
ax2.fill_between(
    time_steps,
    accident_rate - accident_rate_std,
    accident_rate + accident_rate_std,
    color="green",
    alpha=0.2,
)

ax1.set_xlabel(xlabel_value, fontsize=12)
ax1.set_ylabel(ylabel_value_1, fontsize=12)
ax1.set_xticks(xticks_values)
ax1.set_yticks(yticks_values_1)
ax1.set_xlim(xlim_values)
ax1.set_ylim(ylim_values_1)
ax1.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax1.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax1.set_facecolor("#f9f9f9")

# Plotting on the second subplot
ax2.plot(
    time_steps, passenger_count, "*-", color="red", label="Passenger Count (Exponential)"
)
ax2.fill_between(
    time_steps,
    passenger_count - passenger_count_std,
    passenger_count + passenger_count_std,
    color="red",
    alpha=0.2,
)
ax2.set_xlabel(xlabel_value, fontsize=12)
ax2.set_ylabel(ylabel_value_2, fontsize=12)
ax2.set_xticks(xticks_values)
ax2.set_yticks(yticks_values_2)
ax2.set_xlim(xlim_values)
ax2.set_ylim(ylim_values_2)
ax2.legend(loc="upper center", frameon=False, ncol=2, bbox_to_anchor=(0.5, 1.15))
ax2.grid(
    True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5
)
ax2.set_facecolor("#f9f9f9")

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig('line_41.pdf', bbox_inches='tight')
