# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.lines as mlines

# ===================
# Part 2: Data Preparation
# ===================
# Data for the plot
marketing_subjects = ["Digital Marketing", "Business Analytics", "E-commerce"]
marketing_out_start = [7.5, 8.0, 7.8]
marketing_out_group_bias = [-2.0, -1.5, -3.3]
marketing_in_start = [6.9, 7.2, 7.0]
marketing_in_group_bias = [+2.3, +1.8, +1.5]
marketing_labels = ["Out-group bias\n(Business Analytics)", "In-group bias\n(Business Analytics)"]

tech_subjects = ["Fintech", "Cryptocurrency", "Quantum Computing"]
tech_out_start = [8.7, 8.5, 7.8]
tech_out_group_bias = [-2.8, -3.0, -3.1]
tech_in_start = [7.8, 7.9, 7.2]
tech_in_group_bias = [+1.7, +2.0, +1.4]
tech_labels = ["Out-group bias\n(Cryptocurrency)", "In-group bias\n(Cryptocurrency)"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure with two subplots
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 5))

# Set the y-axis offsets to be in the middle of each grid
offset = 0.5

# First subplot (marketing_subjects)
for i, subject in enumerate(marketing_subjects):
    # Out-group bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(marketing_out_start[i], i + offset * 3 / 2),
        xytext=(marketing_out_start[i] + marketing_out_group_bias[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color="dodgerblue"),
    )
    ax1.scatter(
        [marketing_out_start[i], marketing_out_start[i] + marketing_out_group_bias[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{marketing_out_group_bias[i]:.2f}",
        (marketing_out_start[i] + marketing_out_group_bias[i], i + offset * 1.75),
        color="dodgerblue",
        ha="right",
        va="center",
    )

    # In-group bias line with arrow and black dots at start and end
    ax1.annotate(
        "",
        xy=(marketing_in_start[i], i + offset / 2),
        xytext=(marketing_in_start[i] + marketing_in_group_bias[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color="gray"),
    )
    ax1.scatter(
        [marketing_in_start[i], marketing_in_start[i] + marketing_in_group_bias[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax1.annotate(
        f"{marketing_in_group_bias[i]:.2f}",
        (marketing_in_start[i] + marketing_in_group_bias[i], i + offset * 0.75),
        color="gray",
        ha="left",
        va="center",
    )

# Second subplot (tech_subjects)
for i, subject in enumerate(tech_subjects):
    ax2.annotate(
        "",
        xy=(tech_out_start[i], i + offset * 3 / 2),
        xytext=(tech_out_start[i] + tech_out_group_bias[i], i + offset * 3 / 2),
        arrowprops=dict(arrowstyle="<-", color="dodgerblue"),
    )
    ax2.scatter(
        [tech_out_start[i], tech_out_start[i] + tech_out_group_bias[i]],
        [i + offset * 3 / 2, i + offset * 3 / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{tech_out_group_bias[i]:.2f}",
        (tech_out_start[i] + tech_out_group_bias[i], i + offset * 1.75),
        color="dodgerblue",
        ha="right",
        va="center",
    )

    ax2.annotate(
        "",
        xy=(tech_in_start[i], i + offset / 2),
        xytext=(tech_in_start[i] + tech_in_group_bias[i], i + offset / 2),
        arrowprops=dict(arrowstyle="<-", color="gray"),
    )
    ax2.scatter(
        [tech_in_start[i], tech_in_start[i] + tech_in_group_bias[i]],
        [i + offset / 2, i + offset / 2],
        color="black",
        s=10,
    )
    ax2.annotate(
        f"{tech_in_group_bias[i]:.2f}",
        (tech_in_start[i] + tech_in_group_bias[i], i + offset * 0.75),
        color="gray",
        ha="left",
        va="center",
    )

# Set y-axis limits
ax1.set_ylim(0, len(marketing_subjects))
ax2.set_ylim(0, len(tech_subjects))

# Set x-axis limits uniformly
ax1.set_xlim(2, 12)
ax2.set_xlim(2, 12)

# Adjust the y-axis tick positions
ax1.set_yticks([i + offset for i in range(len(marketing_subjects))])
ax1.set_yticklabels(marketing_subjects)
ax2.set_yticks([i + offset for i in range(len(tech_subjects))])
ax2.set_yticklabels(tech_subjects)
ax2.yaxis.tick_right()
ax2.yaxis.set_label_position("right")

# Offset grid lines on the y-axis
ax1.set_yticks([i for i in range(len(marketing_subjects))], minor=True)
ax2.set_yticks([i for i in range(len(tech_subjects))], minor=True)
ax1.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")
ax2.yaxis.grid(True, which="minor", linewidth=0.5, alpha=0.7, color="black")

# Add x-axis grid lines and set gap is 1
ax1.xaxis.set_major_locator(plt.MultipleLocator(1))
ax2.xaxis.set_major_locator(plt.MultipleLocator(1))
ax1.grid(axis="x", linestyle="--", linewidth=0.5)
ax2.grid(axis="x", linestyle="--", linewidth=0.5)

# Create arrow-shaped legend entries with a line that aligns with the arrowhead
dodgerblue_arrow = mlines.Line2D(
    [],
    [],
    color="dodgerblue",
    marker=">",
    linestyle="-",
    markersize=8,
    label=marketing_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
gray_arrow = mlines.Line2D(
    [],
    [],
    color="gray",
    marker=">",
    linestyle="-",
    markersize=8,
    label=marketing_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[dodgerblue_arrow, gray_arrow], bbox_to_anchor=(0.45, 0), ncol=2)

dodgerblue_arrow = mlines.Line2D(
    [],
    [],
    color="dodgerblue",
    marker=">",
    linestyle="-",
    markersize=8,
    label=tech_labels[0],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
gray_arrow = mlines.Line2D(
    [],
    [],
    color="gray",
    marker=">",
    linestyle="-",
    markersize=8,
    label=tech_labels[1],
    linewidth=2,
    markeredgewidth=2,
    markevery=(1, 1),
)
fig.legend(handles=[dodgerblue_arrow, gray_arrow], bbox_to_anchor=(0.85, 0), ncol=2)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save the plot
plt.tight_layout()
plt.savefig('quiver_6.pdf', bbox_inches='tight')