# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
# Setting a random seed for reproducibility
import numpy as np; np.random.seed(0)
# Generate mean energy consumption for 5 years for 2 regions
mean_consumption_southern = np.linspace(200, 350, 5)
mean_consumption_northern = np.linspace(320, 280, 5)
# Smaller standard deviations for realistic energy consumption clustering
standard_deviations = [50] * 5

# Generate data for Southern and Northern regions
southern_consumption = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_consumption_southern, standard_deviations)
]
northern_consumption = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_consumption_northern, standard_deviations)
]
positions_southern = np.array(range(1, len(southern_consumption) + 1)) - 0.2
positions_northern = np.array(range(1, len(northern_consumption) + 1)) + 0.2
legend_labels = ["Southern China", "Northern China"]
xlabel = "Year"
ylabel = "Energy Consumption (GWh)"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["2019", "2020", "2021", "2022", "2023"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and axis with specified dimensions
fig, ax = plt.subplots(figsize=(8, 7))

# Violin plot width
violin_width = 0.35

# Create the violin plot with adjusted positions
parts_southern = ax.violinplot(
    southern_consumption, positions=positions_southern, widths=violin_width, showmeans=False, showmedians=True
)
parts_northern = ax.violinplot(
    northern_consumption, positions=positions_northern, widths=violin_width, showmeans=False, showmedians=True
)

# Customizing the colors of the violin plot
for pc in parts_southern["bodies"]:
    pc.set_facecolor("palevioletred")  # Southern region color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

for pc in parts_northern["bodies"]:
    pc.set_facecolor("aquamarine")  # Northern region color
    pc.set_edgecolor("black")
    pc.set_alpha(0.7)

# Customizing the median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp = parts_southern[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

    vp = parts_northern[partname]
    vp.set_edgecolor("black")
    vp.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp.set_visible(False)  # Hide caps

# Adding statistics annotations for both regions
for i in range(len(southern_consumption)):
    # Southern region statistics
    quartile1, median, quartile3 = np.percentile(southern_consumption[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(southern_consumption[i][southern_consumption[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(southern_consumption[i][southern_consumption[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions_southern[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(
        median,
        positions_southern[i] - 0.025,
        positions_southern[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_southern[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1
    )

    # Northern region statistics
    quartile1, median, quartile3 = np.percentile(northern_consumption[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(northern_consumption[i][northern_consumption[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(northern_consumption[i][northern_consumption[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions_northern[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(
        median,
        positions_northern[i] - 0.025,
        positions_northern[i] + 0.025,
        color="white",
        linestyle="-",
        lw=1,
        zorder=3,
    )
    ax.vlines(
        positions_northern[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1
    )

# Change the border color to grey
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove small ticks beside the numbers on the x and y axes
ax.tick_params(axis="both", which="both", length=0)

# Adding the corrected legend
ax.legend(
    [parts_southern["bodies"][0], parts_northern["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting the title and labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting the x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(
    True, linestyle="-", linewidth=0.7, color="grey", zorder=0
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size to match original image's dimensions
fig.set_size_inches(7, 5)  # Width, Height in inches

# Adjust layout for better fit
plt.tight_layout()

# Display the plot
plt.savefig('violin_12.pdf', bbox_inches='tight')