# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate mean values for energy output (in kWh)
mean_values_solar_a = np.linspace(25, 35, 5)
mean_values_solar_b = np.linspace(40, 20, 5)

# Standard deviations
standard_deviations = [5] * 5

# Generate synthetic data for Solar Panels A
data_solar_a = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_solar_a, standard_deviations)
]

# Generate synthetic data for Solar Panels B
data_solar_b = [
    np.random.normal(loc=mean, scale=std, size=50)
    for mean, std in zip(mean_values_solar_b, standard_deviations)
]

# Set positions for violins
positions_solar_a = np.array(range(1, len(data_solar_a) + 1)) - 0.2
positions_solar_b = np.array(range(1, len(data_solar_b) + 1)) + 0.2

# Labels and ticks
legend_labels = ["Solar Panels A", "Solar Panels B"]
xlabel = "Weather Conditions"
ylabel = "Energy Output (kWh)"
xticks = [1, 2, 3, 4, 5]
xtickslabels = ["Sunny", "Partly Cloudy", "Cloudy", "Rainy", "Stormy"]
# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig, ax = plt.subplots(figsize=(8, 7))

violin_width = 0.35

# Create violin plots
parts_solar_a = ax.violinplot(
    data_solar_a, positions=positions_solar_a, widths=violin_width, showmeans=False, showmedians=True
)
parts_solar_b = ax.violinplot(
    data_solar_b, positions=positions_solar_b, widths=violin_width, showmeans=False, showmedians=True
)

# Customizing colors
for pc in parts_solar_a["bodies"]:
    pc.set_facecolor("cornsilk")
    pc.set_edgecolor("black")
    pc.set_alpha(1)

for pc in parts_solar_b["bodies"]:
    pc.set_facecolor("powderblue")
    pc.set_edgecolor("black")
    pc.set_alpha(1)

# Customizing median lines and removing caps
for partname in ("cbars", "cmins", "cmaxes", "cmedians"):
    vp_solar_a = parts_solar_a[partname]
    vp_solar_b = parts_solar_b[partname]
    vp_solar_a.set_edgecolor("black")
    vp_solar_b.set_edgecolor("black")
    vp_solar_a.set_linewidth(1)
    vp_solar_b.set_linewidth(1)
    if partname in ("cmins", "cmaxes", "cmedians"):
        vp_solar_a.set_visible(False)  # Hide caps
        vp_solar_b.set_visible(False)  # Hide caps

# Adding statistical annotations
for i in range(len(data_solar_a)):
    quartile1, median, quartile3 = np.percentile(data_solar_a[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_solar_a[i][data_solar_a[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_solar_a[i][data_solar_a[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions_solar_a[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(median, positions_solar_a[i] - 0.025, positions_solar_a[i] + 0.025, color="white", linestyle="-", lw=1, zorder=3)
    ax.vlines(positions_solar_a[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1)

    quartile1, median, quartile3 = np.percentile(data_solar_b[i], [25, 50, 75])
    iqr = quartile3 - quartile1
    lower_whisker = np.min(data_solar_b[i][data_solar_b[i] >= quartile1 - 1.5 * iqr])
    upper_whisker = np.max(data_solar_b[i][data_solar_b[i] <= quartile3 + 1.5 * iqr])
    ax.vlines(positions_solar_b[i], quartile1, quartile3, color="black", linestyle="-", lw=4)
    ax.hlines(median, positions_solar_b[i] - 0.025, positions_solar_b[i] + 0.025, color="white", linestyle="-", lw=1, zorder=3)
    ax.vlines(positions_solar_b[i], lower_whisker, upper_whisker, color="black", linestyle="-", lw=1)

# Customize borders
for spine in ax.spines.values():
    spine.set_edgecolor("grey")

# Remove ticks
ax.tick_params(axis="both", which="both", length=0)

# Adding legend
ax.legend(
    [parts_solar_a["bodies"][0], parts_solar_b["bodies"][0]],
    legend_labels,
    loc="upper center",
    bbox_to_anchor=(0.5, 1.1),
    ncol=2,
)

# Setting labels
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Setting x-axis labels
ax.set_xticks(xticks)
ax.set_xticklabels(xtickslabels)

# Enabling y-axis grid lines
ax.yaxis.grid(
    True, linestyle="-", linewidth=0.7, color="grey", zorder=0
)

# ===================
# Part 4: Saving Output
# ===================
fig.set_size_inches(7, 5)

plt.tight_layout()
plt.savefig("violin_13.pdf", bbox_inches="tight")