
# ===================
# Part 1: Importing Libraries
# ===================
import math
import matplotlib.pyplot as plt

from matplotlib.collections import PolyCollection

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)

# Function to create polygons under the graph
def polygon_under_graph(x, y):
    """
    Construct the vertex list which defines the polygon filling the space under
    the (x, y) line graph. This assumes x is in ascending order.
    """
    return [(x[0], 0.0), *zip(x, y), (x[-1], 0.0)]

# New dataset for communication networks
nodes = np.linspace(0.0, 10.0, 31)
packet_sizes = [64, 128, 256]
latency = lambda size, nodes: (np.sin(nodes / 2) + 1) / size  # Simulated latency data
packet_loss = lambda size, nodes: (np.cos(nodes / 2) + 1) * 0.02  # Simulated packet loss rate

# Create polygons to represent data
verts = [
    polygon_under_graph(nodes, latency(size, nodes) + packet_loss(size, nodes))
    for size in packet_sizes
]

# Labels
xlabel = "Node"
ylabel = "Packet Size (bytes)"
zlabel = "Latency + Packet Loss"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig = plt.figure(figsize=(10, 8))
ax = fig.add_subplot(projection="3d")
facecolors = plt.cm.viridis(np.linspace(0, 1, len(verts)))

poly = PolyCollection(verts, facecolors=facecolors, alpha=0.7)
ax.add_collection3d(poly, zs=packet_sizes, zdir="y")

ax.set(
    xlim=(0, 10),
    ylim=(min(packet_sizes), max(packet_sizes) + 1),
    zlim=(0, 0.1),
    xlabel=xlabel,
    ylabel=ylabel,
    zlabel=zlabel,
    title="Network Performance: Latency and Packet Loss"
)

ax.set_yticks(packet_sizes)
ax.grid(True)

# Additional styling adjustments
ax.view_init(elev=20., azim=-35)

plt.tight_layout()
plt.savefig("3d_62.pdf", bbox_inches="tight")
