
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.collections import PolyCollection

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)

# Function to create polygon under graph
def polygon_under_graph(x, y):
    return [(x[0], 0.0), *zip(x, y), (x[-1], 0.0)]

# Data for bar chart
years = [2014, 2015, 2016, 2017, 2018, 2019, 2020, 2021, 2022, 2023]
streaming = [1.8, 2.3, 3.1, 4.0, 5.6, 6.3, 9.1, 10.2, 11.0, 12.5]
cinema = [10.5, 10.8, 11.2, 11.5, 11.0, 10.8, 3.0, 7.5, 8.0, 7.8]

# Data for distribution graph
x = np.linspace(0.0, 10.0, 31)
content_types = range(1, 4)
exp = np.exp
verts = [
    polygon_under_graph(x, exp(-0.5 * (x - t) ** 2)) for t in content_types
]  # Gaussian distributions

xlabels = ["Year", "Years Since Release"]
ylabels = ["Industry", "Content Type"]
zlabels = ["Revenue (Billion USD)", "Engagement Rate"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
facecolors = plt.colormaps["viridis"](np.linspace(0, 1, len(verts)))

# Initialize figure and axes
fig = plt.figure(figsize=(14, 8))
ax1 = fig.add_subplot(121, projection="3d")  # 3D bar chart
ax2 = fig.add_subplot(122, projection="3d")  # 3D distribution graph

# Data for bar chart
ax1.bar(years, streaming, zs=0, zdir="y", color="#ab63fa", alpha=0.8)
ax1.bar(years, cinema, zs=1, zdir="y", color="#19d3f3", alpha=0.8)

# Set labels and ticks for bar chart
ax1.set_xlabel(xlabels[0])
ax1.set_ylabel(ylabels[0])
ax1.set_zlabel(zlabels[0])
ax1.set_yticks([0, 1])
ax1.set_yticklabels(["Streaming", "Cinema"])

# Add polygons to the plot
poly = PolyCollection(verts, facecolors=facecolors, alpha=0.7)
ax2.add_collection3d(poly, zs=content_types, zdir="y")

# Set labels and limits for distribution graph
ax2.set(
    xlim=(0, 10),
    ylim=(1, 4),
    zlim=(0, 1),
    xlabel=xlabels[1],
    ylabel=ylabels[1],
    zlabel=zlabels[1],
)
ax2.set_yticks([1, 2, 3])

# ===================
# Part 4: Saving Output
# ===================
# Adjust the layout and save the figure
plt.tight_layout()
plt.suptitle('Entertainment Trends and Engagement Analysis', fontsize=16)
fig.subplots_adjust(top=1)
ax1.set_box_aspect(aspect=None, zoom=0.9)
ax2.set_box_aspect(aspect=None, zoom=0.8)
plt.savefig("3d_63.pdf", bbox_inches="tight")
