
# ===================
# Part 1: Importing Libraries
# ===================
import math
import matplotlib.pyplot as plt
from matplotlib.collections import PolyCollection

# ===================
# Part 2: Data Preparation
# ===================
# Set a random seed for reproducibility
import numpy as np

np.random.seed(42)

def polygon_under_graph(x, y):
    """
    Construct the vertex list which defines the polygon filling the space under
    the (x, y) line graph. This assumes x is in ascending order.
    """
    return [(x[0], 0.0), *zip(x, y), (x[-1], 0.0)]

# Define the x-axis data (Centuries)
x = np.arange(1, 11)  # Centuries: 1 to 10
regions = range(1, 4)  # Three regions

# New function to simulate population growth over centuries
def population_growth(x, region):
    # Simulate exponential population growth for different regions
    return region * np.exp(0.3 * x - 1.5) / 10  # population in billions

# Generate vertices for polygons
verts = [polygon_under_graph(x, population_growth(x, region)) for region in regions]

# Extracted variables
x_label = "Centuries"
y_label = "Regions"
z_label = "Population (Billions)"
title = "Population Growth Over Centuries by Region"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and 3D axis
yticks = [1, 2, 3]
xlim = (1, 10)
ylim = (1, 4)
zlim = (0, 2.5)
aspect_ratio = [2, 1, 1]
view_elev = 30
view_azim = 210

fig = plt.figure(figsize=(12, 9))
ax = fig.add_subplot(projection="3d")

# Define the face colors
facecolors = plt.get_cmap("autumn")(np.linspace(0, 1, len(verts)))

# Create a PolyCollection object
poly = PolyCollection(verts, facecolors=facecolors, alpha=0.7)
ax.add_collection3d(poly, zs=regions, zdir="y")

# Set the axis labels and limits
ax.set(xlim=xlim, ylim=ylim, zlim=zlim, xlabel=x_label, ylabel=y_label, zlabel=z_label)

# Add ticks and title
ax.set_yticks(yticks)
ax.set_title(title, pad=20)

# Adjust the aspect ratio and view angle
ax.set_box_aspect(aspect_ratio)
ax.view_init(elev=view_elev, azim=view_azim)

# ===================
# Part 4: Saving Output
# ===================
# Adjust the layout and save the figure
plt.tight_layout()
plt.savefig("3d_80.pdf", bbox_inches="tight")
