
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Sample data for Sales Performance Analysis
quarters = ["Q1", "Q2", "Q3", "Q4"]
region_A = [5000, 7000, 8000, 9000]
region_B = [4500, 6800, 7500, 8500]
region_C = [5200, 6900, 7700, 8800]
region_D = [4800, 7200, 7800, 9100]
growth_rates = [0.10, 0.15, 0.12, 0.08]

# Text labels
labels = ["Region A", "Region B", "Region C", "Region D"]
xlabel = "Quarter"
ylabel1 = "Sales (USD)"
ylabel2 = "Growth Rate (%)"
legend_title = "Regions"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axis
fig, ax1 = plt.subplots(figsize=(10, 6))

# Bar plot
bar_width = 0.2
index = np.arange(len(quarters))
ax1.bar(index, region_A, bar_width, label=labels[0], color="royalblue")
ax1.bar(index + bar_width, region_B, bar_width, label=labels[1], color="seagreen")
ax1.bar(index + 2 * bar_width, region_C, bar_width, label=labels[2], color="slateblue")
ax1.bar(index + 3 * bar_width, region_D, bar_width, label=labels[3], color="darkorange")

# Line plot for growth rates
ax2 = ax1.twinx()
ax2.plot(
    quarters,
    growth_rates,
    color="crimson",
    marker="s",
    linestyle="--",
    linewidth=2,
    markersize=6,
)

# Annotate growth rates
for i, rate in enumerate(growth_rates):
    ax2.annotate(
        f"{rate*100:.0f}%",
        (index[i] + bar_width * 1.5, rate),
        textcoords="offset points",
        xytext=(0, 10),
        ha="center",
        color="crimson",
    )

# Set labels and title
ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabel1)
ax2.set_ylabel(ylabel2)

# Set x-axis tick labels
ax1.set_xticks(index + bar_width * 1.5)
ax1.set_xticklabels(quarters)

# Add legend
ax1.legend(loc="upper left", ncol=2, bbox_to_anchor=(0.12, -0.1), title=legend_title)
ax1.grid(axis="y", linestyle="--", alpha=0.7)
ax1.set_axisbelow(True)

# set ax2.yticklabels to be percentage
ax2.set_yticklabels([f"{x*100:.0f}%" for x in ax2.get_yticks()])

# ===================
# Part 4: Saving Output
# ===================
# Show plot with tight layout
plt.tight_layout()
# Save the plot as a PDF file
plt.savefig("CB_104.pdf", bbox_inches="tight")
