
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)  # Setting a seed for reproducibility

# Generating random data to represent the average monthly temperatures (in Celsius) for different cities
data_city_A = np.random.normal(15, 5, 12)  # City A
data_city_B = np.random.normal(20, 3, 12)  # City B
data_city_C = np.random.normal(10, 4, 12)  # City C
data_reference = np.random.normal(18, 2, 12)  # Reference City

# Packing the data into a list
data = [data_city_A, data_city_B, data_city_C, data_reference]
legend_labels = ["City A", "City B", "City C", "Reference City"]
line_label = "Median Monthly Temperature"
ylabel = "Temperature (°C)"
xlabel = "Cities"
title = "Boxplot of Average Monthly Temperatures"
output_filename = "CB_27_5.pdf"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating a box plot
fig, ax1 = plt.subplots(figsize=(10, 6))
bp = ax1.boxplot(data, patch_artist=True, notch=False, showfliers=False, positions=[1, 2, 3, 4])

# Customizing the boxplot colors
colors = ["#69b3a2", "#ff6f61", "#76c7c0", "#f4a460"]
for patch, color in zip(bp["boxes"], colors):
    patch.set_facecolor(color)
for median in bp["medians"]:
    median.set(color="black", zorder=2)

# Extracting medians for the line graph
medians = [np.median(d) for d in data]

# Creating the line graph on the same axes
ax1.plot(
    [1, 2, 3, 4],
    medians,
    "-o",
    color="black",
    label=line_label,
    ms=10,
    markerfacecolor="#ff1493"
)

# Setting legend for the boxplot
legend_patches = [mpatches.Patch(color=color, label=label) for color, label in zip(colors, legend_labels)]
ax1.legend(
    handles=legend_patches + [mpatches.Patch(color="#ff1493", label=line_label)],
    loc="upper right"
)

# Setting labels for the x-axis
ax1.set_xticks([1, 2, 3, 4])
ax1.set_xticklabels(legend_labels)

# Setting the y-axis and x-axis labels
ax1.set_ylabel(ylabel)
ax1.set_xlabel(xlabel)

# Setting y-axis limits and adding grid lines
ax1.set_ylim(0, 30)
ax1.yaxis.grid(True, which="major", linestyle="--", color="grey", alpha=0.5)

# Removing top and right spines for aesthetics
ax1.spines["top"].set_visible(False)
ax1.spines["right"].set_visible(False)

# Setting the title
plt.title(title)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_110.pdf", bbox_inches="tight")
