
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)  # Setting a seed for reproducibility

# Generating random data to represent the length of reign (in years) for different historical empires
data_roman_empire = np.random.normal(500, 50, 100)  # Roman Empire
data_ottoman_empire = np.random.normal(600, 30, 100)  # Ottoman Empire
data_british_empire = np.random.normal(400, 40, 100)  # British Empire
data_mongol_empire = np.random.normal(200, 20, 100)  # Mongol Empire

# Packing the data into a list
data = [data_roman_empire, data_ottoman_empire, data_british_empire, data_mongol_empire]
legend_labels = ["Roman Empire", "Ottoman Empire", "British Empire", "Mongol Empire"]
line_label = "Median Length of Reign"
ylabel = "Length of Reign (Years)"
xlabel = "Historical Empires"
label = "Median Length of Reign"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating a box plot
fig, ax1 = plt.subplots(figsize=(10, 6))
bp = ax1.boxplot(
    data, patch_artist=True, notch=False, showfliers=False, positions=[1, 2, 3, 4]
)

# Customizing the boxplot colors
colors = ["#8c510a", "#d8b365", "#c7eae5", "#5ab4ac"]  # Earthy, historical tones
for patch, color in zip(bp["boxes"], colors):
    patch.set_facecolor(color)
for median in bp["medians"]:
    median.set(color="black", zorder=2)

# Extracting medians for the line graph
medians = [np.median(d) for d in data]

# Creating the line graph on the same axes
ax1.plot(
    [1, 2, 3, 4],
    medians,
    "-o",
    color="black",
    label=label,
    ms=10,
    markerfacecolor="#e66101",
)

# Setting legend for the boxplot
legend_patches = [
    mpatches.Patch(color=color, label=label)
    for color, label in zip(colors, legend_labels)
]
ax1.legend(
    handles=legend_patches + [mpatches.Patch(color="#e66101", label=line_label)],
    loc="upper right",
)

# Setting labels for the x-axis
ax1.set_xticklabels(legend_labels)

# Setting the y-axis label
ax1.set_ylabel(ylabel)
ax1.set_xlabel(xlabel)

# Setting y-axis limits and adding grid lines
ax1.set_ylim(100, 700)
ax1.yaxis.grid(True, which="major", linestyle="--", color="grey", alpha=0.5)

# Removing top and right spines for aesthetics
ax1.spines["top"].set_visible(False)
ax1.spines["right"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_111.pdf", bbox_inches="tight")
