
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import norm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# New Sample data for agriculture
organic_yields = np.random.normal(loc=25, scale=5, size=500)
conventional_yields = np.random.normal(loc=30, scale=5, size=500)

# Labels and Plot Types
ax1_label = "Organic Farming"
ax2_label = "Conventional Farming"
ax1_text = r"Average Organic"
ax2_text = r"Average Conventional"

# Axes Limits and Labels
xlabel_value = "Crop Yield (tons/acre)"
ylabel_value = "Frequency"
xticks_values = [10, 15, 20, 25, 30, 35, 40]
xlim_values = [9, 41]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axis
fig, ax = plt.subplots(
    figsize=(10, 8)
)

# Plot histograms
ax.hist(organic_yields, bins=15, density=True, alpha=0.7, color="#69b3a2", edgecolor='black', label=ax1_label)
ax.hist(conventional_yields, bins=15, density=True, alpha=0.7, color="#404080", edgecolor='black', label=ax2_label)

# Plot normal distributions
xmin, xmax = ax.get_xlim()
x = np.linspace(xmin, xmax, 100)
p_organic = norm.pdf(x, np.mean(organic_yields), np.std(organic_yields))
p_conventional = norm.pdf(x, np.mean(conventional_yields), np.std(conventional_yields))
ax.plot(x, p_organic, color="#1aaf54", linestyle='-', linewidth=3)
ax.plot(x, p_conventional, color="#a52a2a", linestyle='--', linewidth=3)

# Add dashed lines at mean
ax.axvline(np.mean(organic_yields), color="black", linestyle="dashed")
ax.axvline(np.mean(conventional_yields), color="black", linestyle="dashed")

# Add text labels for dashed lines
ax.text(
    np.mean(organic_yields) + 1,
    ax.get_ylim()[1] - 0.02,
    ax1_text,
    ha="center",
    va="top",
    fontsize=14,
)
ax.text(
    np.mean(conventional_yields) + 1,
    ax.get_ylim()[1] - 0.04,
    ax2_text,
    ha="center",
    va="top",
    fontsize=14,
)

# Set labels and title
ax.set_xlabel(xlabel_value)
ax.set_ylabel(ylabel_value)
ax.set_title('Comparison of Crop Yields Between Organic and Conventional Farming')
ax.set_xticks(xticks_values)
ax.set_xlim(xlim_values)
ax.grid(True, linestyle='--', alpha=0.7)

# Add legend
ax.legend()

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save plot
plt.tight_layout()
plt.savefig("CB_31.pdf", bbox_inches="tight")
