
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.lines import Line2D

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generate data related to society - happiness level and time spent on hobbies
happiness_level = np.random.rand(100)
time_spent_on_hobbies = 0.6 * happiness_level + np.random.rand(100) * 0.3

# Add a line for visualization purposes
line_x = [0.2, 0.6, 0.7, 0.9, 1.0]
line_y = [0.3, 0.55, 0.65, 0.75, 0.85]
diagonal_line = [[0, 1], [0, 1]]

scatters_name = "Happiness vs. Hobbies"
bin_edges_name = "Bin Edges"
calibration_curve_name = "Trend Line"

line_label = "Trend Line"
xlabel = "Happiness Level"
ylabel = "Time Spent on Hobbies"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Draw the scatter plot and marginals.

# Start with a square Figure.
fig = plt.figure(figsize=(6, 6))
gs = fig.add_gridspec(
    2,
    1,
    height_ratios=(1, 4),
    left=0.1,
    right=0.9,
    bottom=0.1,
    top=0.9,
    wspace=0.0,
    hspace=0.0,
)
# Create the Axes.
ax = fig.add_subplot(gs[1, 0])
ax_histx = fig.add_subplot(gs[0, 0], sharex=ax)

# no labels
ax_histx.tick_params(axis="x", labelbottom=False)

# the scatter plot:
sc = ax.scatter(happiness_level, time_spent_on_hobbies, c=happiness_level, cmap="Blues", s=70, edgecolor="black", alpha=0.75)

# add a line on ax
ax.plot(
    line_x,
    line_y,
    color="navy",
    linewidth=2,
    marker="o",
    markersize=8,
    label=line_label,
)

# draw a diagonal line
ax.plot(diagonal_line[0], diagonal_line[1], color="darkgrey", linestyle="--")

# now determine nice limits by hand:
binwidth = 0.05
xymax = max(np.max(np.abs(happiness_level)), np.max(np.abs(time_spent_on_hobbies)))
lim = (int(xymax / binwidth) + 1) * binwidth
bins = np.arange(0, lim + binwidth, binwidth)
ax_histx.hist(happiness_level, bins=bins, color="#87ceeb", edgecolor="black")

# remove the y-axis labels
ax_histx.set_yticks([])

# remove the top, left, and right spines
ax_histx.spines["top"].set_visible(False)
ax_histx.spines["left"].set_visible(False)
ax_histx.spines["right"].set_visible(False)

ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

ax.set_xticks([0.00, 0.25, 0.50, 0.75, 1.00])
ax.set_yticks([0.00, 0.25, 0.50, 0.75, 1.00])
ax.set_xlim(-0.05, 1.05)
ax.set_ylim(-0.05, 1.05)

# draw bin edges
for i in bins:
    ax.axvline(i, color="gray", linewidth=0.5, linestyle="--", zorder=0, alpha=0.5)

# Create an axes on the top side of ax_heatmap_top for the colorbar.
ax_colorbar = fig.add_axes(
    [ax.get_position().width * 1.15, 0.1, 0.05, ax.get_position().y1 * 0.86]
)
cbar = plt.colorbar(sc, cax=ax_colorbar, orientation="vertical")

# Create a Line2D instance for bin edges
bin_edge_line = Line2D([0], [0], color="gray", linewidth=0.5, linestyle="--")

# Create a Line2D instance for calibration curve
calibration_curve_line = Line2D(
    [0],
    [0],
    color="navy",
    linewidth=2,
    marker="o",
    markersize=8,
)

# Update legend to include the new Line2D instances
ax.legend(
    handles=[sc, bin_edge_line, calibration_curve_line],
    labels=[scatters_name, bin_edges_name, calibration_curve_name],
    loc="upper left",
)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_39.pdf", bbox_inches="tight")
