
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Fixing random state for reproducibility
# Simulated environmental data
temperature = np.random.uniform(low=14, high=26, size=300)
co2_concentration = 0.7 * temperature + np.random.normal(loc=20, scale=5, size=300)

xlabel = "Temperature (°C)"
ylabel = "CO₂ Concentration (ppm)"
title = "Scatter Plot of Temperature vs CO₂ Concentration"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Start with a square Figure.
fig = plt.figure(figsize=(6, 6))
# Add a gridspec with two rows and two columns and a ratio of 1 to 4 between
# the size of the marginal axes and the main axes in both directions.
# Also adjust the subplot parameters for a square plot.
gs = fig.add_gridspec(
    2,
    2,
    width_ratios=(4, 1),
    height_ratios=(1, 4),
    left=0.1,
    right=0.9,
    bottom=0.1,
    top=0.9,
    wspace=0.0,
    hspace=0.0,
)
# Create the Axes.
ax = fig.add_subplot(gs[1, 0])
ax_histx = fig.add_subplot(gs[0, 0], sharex=ax)
ax_histy = fig.add_subplot(gs[1, 1], sharey=ax)

# Draw the scatter plot and marginals.
# no labels
ax_histx.tick_params(axis="x", labelbottom=False)
ax_histy.tick_params(axis="y", labelleft=False)

# the scatter plot:
ax.scatter(temperature, co2_concentration, color="#2ca02c", alpha=0.6, edgecolor="black", marker='o')

# now determine nice limits by hand:
binwidth = 1
xymax = max(np.max(np.abs(temperature)), np.max(np.abs(co2_concentration)))
lim = (int(xymax / binwidth) + 1) * binwidth

bins = np.arange(0, lim + binwidth, binwidth)
ax_histx.hist(temperature, bins=bins, color="lightgreen", edgecolor="#2ca02c")
ax_histy.hist(
    co2_concentration, bins=bins, orientation="horizontal", color="lightgreen", edgecolor="#2ca02c"
)

# remove the y-axis labels
ax_histx.set_yticks([])
ax_histy.set_xticks([])

# add grid lines to the scatter plot for better readability
ax.grid(True, linestyle='--', linewidth=0.5)

# set axis labels
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_title(title, fontsize=14, pad=20)

# remove the top, left, and right spines of the histograms
ax_histx.spines["top"].set_visible(False)
ax_histx.spines["left"].set_visible(False)
ax_histx.spines["right"].set_visible(False)
ax_histy.spines["top"].set_visible(False)
ax_histy.spines["bottom"].set_visible(False)
ax_histy.spines["right"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_43.pdf", bbox_inches="tight")

