
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Define parameters for three Gaussian distributions
distributions = {
    "Ancient": {"mean": [25, 0.75], "cov": [[4, 0.5], [0.3, 0.4]]},
    "Medieval": {"mean": [30, 0.85], "cov": [[5, 0.4], [0.2, 0.3]]},
    "Modern": {"mean": [35, 0.7], "cov": [[6, 0.6], [0.4, 0.5]]},
}
keys = ["Ancient", "Medieval", "Modern"]
# Generate samples
samples = {
    category: np.random.multivariate_normal(dist["mean"], dist["cov"], 100)
    for category, dist in distributions.items()
}

xlabel = "Influence Score"
ylabel = "Complexity Index"
title = "Philosophical Eras"
legend_title = "Eras"
savefile = "CB_6_6.pdf"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the plotting grid
fig = plt.figure(figsize=(10, 8))
grid = plt.GridSpec(4, 4, hspace=0, wspace=0)

# Main scatter plot
main_ax = fig.add_subplot(grid[1:, :-1])
colors = {keys[0]: "#1f77b4", keys[1]: "#ff7f0e", keys[2]: "#2ca02c"}
for category, color in colors.items():
    subset = samples[category]
    main_ax.scatter(subset[:, 0], subset[:, 1], c=color, label=category, alpha=0.6, edgecolor='k')
main_ax.set_xlabel(xlabel)
main_ax.set_ylabel(ylabel)

# Top density plot
top_ax = fig.add_subplot(grid[0, :-1], sharex=main_ax)
all_samples = np.concatenate([samples[category] for category in samples], axis=0)
x_min, x_max = all_samples[:, 0].min(), all_samples[:, 0].max()
xs = np.linspace(x_min, x_max, 200)
for category, color in colors.items():
    density = gaussian_kde(samples[category][:, 0])
    top_ax.fill_between(xs, density(xs), alpha=0.6, color=color)

# Right density plot
right_ax = fig.add_subplot(grid[1:, -1], sharey=main_ax)
y_min, y_max = all_samples[:, 1].min(), all_samples[:, 1].max()
ys = np.linspace(y_min, y_max, 200)
for category, color in colors.items():
    density = gaussian_kde(samples[category][:, 1])
    right_ax.fill_betweenx(ys, density(ys), alpha=0.6, color=color)

# Hide the spines
top_ax.spines["top"].set_visible(False)
top_ax.spines["right"].set_visible(False)
top_ax.spines["left"].set_visible(False)
right_ax.spines["top"].set_visible(False)
right_ax.spines["right"].set_visible(False)
right_ax.spines["bottom"].set_visible(False)

# Remove the labels from the top and right axes
top_ax.tick_params(axis="x", which="both", top=False, bottom=False, labelbottom=False)
right_ax.tick_params(axis="y", which="both", left=False, right=False, labelleft=False)

# Remove all ticks from the right axis
top_ax.set_yticks([])
right_ax.set_xticks([])

main_ax.legend(title=legend_title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_47.pdf", bbox_inches="tight")
