
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)  # Different seed for new data

# Data
modes = ["Formal", "Dialectic", "Analytic", "Existential"]
srcc_values = [0.875, 0.842, 0.899, 0.912]
plcc_values = [0.884, 0.858, 0.905, 0.920]
sum_values = [1.759, 1.700, 1.804, 1.832]

# Labels and Titles
labels = ["Relevance Index (SRCC)", "Coherence Index (PLCC)", "Sum (SRCC + PLCC)"]
xlabel = "Modes of Philosophical Discourse"
ylabel = "Index Values"
title = "Philosophical Discourse Mode Comparison"
legend_labels = ["SRCC", "PLCC", "Sum"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Bar plot
fig, ax1 = plt.subplots(figsize=(12, 7))  # Increased figure size for better clarity

bar_width = 0.35
index = np.arange(len(modes))

bar1 = ax1.bar(index, srcc_values, bar_width, label=legend_labels[0], color="#6a5acd")
bar2 = ax1.bar(index + bar_width, plcc_values, bar_width, label=legend_labels[1], color="#4682b4")

# Line plot
ax2 = ax1.twinx()
line = ax2.plot(index + bar_width / 2, sum_values, color="#dc143c", marker="D", linewidth=2, markersize=8, label=legend_labels[2])

# Annotate bars with values
for rect, value in zip(bar1, srcc_values):
    height = rect.get_height()
    ax1.text(rect.get_x() + rect.get_width() / 2, height + 0.005, f"{value:.3f}", ha="center", va="bottom")

for rect, value in zip(bar2, plcc_values):
    height = rect.get_height()
    ax1.text(rect.get_x() + rect.get_width() / 2, height + 0.005, f"{value:.3f}", ha="center", va="bottom")

# Labels, title and legend
ax1.set_xlabel(xlabel, fontsize=12)
ax1.set_ylabel(ylabel, fontsize=12)
ax1.set_title(title, fontsize=14)
ax1.set_xticks(index + bar_width / 2)
ax1.set_xticklabels(modes, fontsize=11)
ax1.set_ylim(0.83, 0.93)
ax1.set_yticks(np.arange(0.83, 0.94, 0.02))
ax2.set_ylim(1.68, 1.84)
ax2.set_yticks(np.arange(1.68, 1.85, 0.02))
ax1.legend(loc="upper left", fontsize=10)
ax2.legend(loc="upper right", fontsize=10)

ax1.yaxis.grid(True)

# ===================
# Part 4: Saving Output
# ===================
# Adjust figure size to match original image's dimensions
fig.set_size_inches(12, 7)

# Show plot
plt.tight_layout()
plt.savefig("CB_55.pdf", bbox_inches="tight")
