
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Generate new data for transportation domain
biking_data = np.random.beta(a=2, b=5, size=1000) * 60  # commute times in minutes
driving_data = np.random.beta(a=5, b=2, size=1000) * 60  # commute times in minutes

# Compute KDE for both datasets
kde_biking = gaussian_kde(biking_data)
kde_driving = gaussian_kde(driving_data)

# Create an array of values for plotting KDE
x_eval = np.linspace(
    min(np.concatenate([biking_data, driving_data])),
    max(np.concatenate([biking_data, driving_data])),
    1000,
)

# Text labels and annotation details
labels = ["Biking", "Driving", "Biking KDE", "Driving KDE"]
p_text = "Mann-Whitney U test p-value = 0.0000"
x_label = "Commute Time (minutes)"
y_label = "Density"
title = "Commute Time Distribution for Different Transport Modes"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size to a comfortable readable size
plt.figure(figsize=(10, 6))

# Plot histograms with custom colors and transparency
plt.hist(
    biking_data, density=True, bins=30, color="#4db8ff", alpha=0.7, label=labels[0]
)
plt.hist(
    driving_data, density=True, bins=30, color="#ffa64d", alpha=0.7, label=labels[1]
)

# Plot KDEs with different line styles and widths
plt.plot(
    x_eval,
    kde_biking(x_eval),
    linestyle="dashed",
    color="blue",
    linewidth=2,
    label=labels[2],
)
plt.plot(
    x_eval,
    kde_driving(x_eval),
    linestyle="dashed",
    color="red",
    linewidth=2,
    label=labels[3],
)

# Add legend
plt.legend()

# Add text annotation for statistical test result
plt.text(40, 0.06, p_text, fontsize=10)

# Set labels, title and other plot details
plt.xlabel(x_label)
plt.ylabel(y_label)
plt.title(title)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_65.pdf", bbox_inches="tight")
