
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate synthetic data for a psychological study: Self-Esteem vs. Happiness
self_esteem = np.random.rand(100) * 0.8 + 0.2
happiness = np.random.rand(100) * 0.8 + 0.2

# Hypothetical trend line representing correlation
trend_x = [0.2, 0.4, 0.6, 0.8, 1.0]
trend_y = [0.3, 0.35, 0.45, 0.55, 0.7]

# Labels and titles
scatter_name = "Participants"
line_name = "Trend Line"
xlabel = "Self-Esteem"
ylabel = "Happiness"
title = "Self-Esteem vs. Happiness Study"
legend_labels = ["Participants", "Trend Line", "Reference Line"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Start with a square Figure.
fig = plt.figure(figsize=(6, 6))
gs = fig.add_gridspec(
    2,
    1,
    height_ratios=(1, 4),
    left=0.1,
    right=0.9,
    bottom=0.1,
    top=0.9,
    wspace=0.0,
    hspace=0.0,
)
# Create the Axes.
ax = fig.add_subplot(gs[1, 0])
ax_histx = fig.add_subplot(gs[0, 0], sharex=ax)

# Draw the scatter plot and marginals.
ax_histx.tick_params(axis="x", labelbottom=False)

# Scatter plot
ax.scatter(self_esteem, happiness, color="#4B9CD3", edgecolor="white", s=50, label=scatter_name)

# Trend line
ax.plot(trend_x, trend_y, color="#2E86C1", linewidth=2, marker="o", markersize=6, label=line_name)

# Reference diagonal line
ax.plot([0, 1], [0, 1], color="#85C1E9", linestyle="--")

# Histogram for the x-axis data (Self-Esteem)
binwidth = 0.05
xymax = max(np.max(np.abs(self_esteem)), np.max(np.abs(happiness)))
lim = (int(xymax / binwidth) + 1) * binwidth
bins = np.arange(0, lim + binwidth, binwidth)
ax_histx.hist(self_esteem, bins=bins, color="#AED6F1", edgecolor="black")

# Remove the y-axis labels
ax_histx.set_yticks([])

# Remove the top, left, and right spines
ax_histx.spines["top"].set_visible(False)
ax_histx.spines["left"].set_visible(False)
ax_histx.spines["right"].set_visible(False)

# Labeling
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)

ax.set_xticks([0.00, 0.25, 0.50, 0.75, 1.00])
ax.set_yticks([0.00, 0.25, 0.50, 0.75, 1.00])
ax.set_xlim(-0.05, 1.05)
ax.set_ylim(-0.05, 1.05)

ax.legend(loc="upper left")

# Add grid for better readability
ax.grid(True)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_89.pdf", bbox_inches="tight")
