
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate data for the business domain
weeks = np.arange(1, 101)
advertising_budget = np.random.rand(100) * 1000 + 500  # Random budget between 500 and 1500
weekly_sales = advertising_budget * 0.75 + np.random.randn(100) * 50  # Simulate sales with some noise
trend_x = np.linspace(500, 1500, 5)
trend_y = trend_x * 0.75

# Labels and legends
scatter_name = "Weekly Data Points"
line_name = "Trend Line"
xlabel = "Advertising Budget (in $)"
ylabel = "Weekly Sales (in $)"
title = "Weekly Sales vs. Advertising Budget"
supertitle = "Business Analysis: Sales Performance"
legend_location = "upper left"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Start with a square Figure.
fig = plt.figure(figsize=(6, 6))
gs = fig.add_gridspec(
    2,
    1,
    height_ratios=(1, 4),
    left=0.1,
    right=0.9,
    bottom=0.1,
    top=0.9,
    wspace=0.0,
    hspace=0.0,
)
# Create the Axes.
ax = fig.add_subplot(gs[1, 0])
ax_histx = fig.add_subplot(gs[0, 0], sharex=ax)

# Draw the scatter plot and marginals.
# no labels
ax_histx.tick_params(axis="x", labelbottom=False)

# Scatter plot:
ax.scatter(advertising_budget, weekly_sales, color="#4CAF50", edgecolor="white", s=50, label=scatter_name)

# Add a trend line on ax
ax.plot(
    trend_x,
    trend_y,
    color="#2196F3",
    linewidth=2,
    marker="o",
    markersize=6,
    label=line_name,
)

# Draw a diagonal reference line
ax.plot([500, 1500], [375, 1125], color="grey", linestyle="--")

# Determine nice limits by hand
binwidth = 100
xymax = max(np.max(np.abs(advertising_budget)), np.max(np.abs(weekly_sales)))
lim = (int(xymax / binwidth) + 1) * binwidth
bins = np.arange(500, lim + binwidth, binwidth)
ax_histx.hist(advertising_budget, bins=bins, color="#A5D6A7", edgecolor="black")

# Remove y-axis labels for histogram
ax_histx.set_yticks([])

# Remove the top, left, and right spines
ax_histx.spines["top"].set_visible(False)
ax_histx.spines["left"].set_visible(False)
ax_histx.spines["right"].set_visible(False)

# Set titles and labels
fig.suptitle(supertitle, fontsize=14)
ax.set_title(title, fontsize=12)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Set ticks and limits for the scatter plot
ax.set_xticks([500, 750, 1000, 1250, 1500])
ax.set_yticks([375, 500, 750, 1000, 1250])
ax.set_xlim(400, 1600)
ax.set_ylim(300, 1300)

# Add legend
ax.legend(loc=legend_location)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_90.pdf", bbox_inches="tight")
