
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Seed for reproducibility
np.random.seed(42)

# Generate time values (simulating hours in a workday)
time = np.linspace(0, 24, 100)

# Simulate stress levels with a sinusoidal pattern plus random noise
stress_levels = 5 + 2 * np.sin(time / 24 * 2 * np.pi) + np.random.normal(0, 0.5, len(time))

# Normalize stress levels for color mapping
normalized_stress_levels = (stress_levels - np.min(stress_levels)) / (np.max(stress_levels) - np.min(stress_levels))

# Creating a custom linear color map from light green to dark blue based on these normalized stress levels
colors = [plt.cm.BuGn_r(stress) for stress in normalized_stress_levels]

# Text parameters
xlabel = "Time of Day (hours)"
ylabel = "Stress Level"
title = "Simulated Stress Levels Over a Workday"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size
plt.figure(figsize=(10, 6))

# Plot the points with a line
plt.plot(time, stress_levels, color="lightgray", linewidth=1, linestyle="--")
plt.scatter(time, stress_levels, c=colors, s=50, edgecolor="black")

# Set axis labels and title
plt.xlabel(xlabel)
plt.ylabel(ylabel)
plt.title(title)

# Add axis lines
plt.axhline(stress_levels.mean(), color="gray", linewidth=0.5, linestyle="--", label="Average Stress Level")
plt.axvline(12, color="gray", linewidth=0.5, linestyle="--", label="Midday")

# Add grid for better readability
plt.grid(True, which="both", linestyle="--", linewidth=0.5)

# Add legend
plt.legend()

# ===================
# Part 4: Saving Output
# ===================
# Display the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("HR_38.pdf", bbox_inches="tight")
