
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate theta values (angles)
theta = np.linspace(0, 2 * np.pi, 120)

# Generate r values (distance from center) with random noise to simulate uneven population distribution
r = 1 + 0.3 * np.random.randn(120)

# Calculate the x and y coordinates using polar to cartesian transformation
x = r * np.cos(theta)
y = r * np.sin(theta)

# Calculate a custom metric for coloring (population density in this case)
pop_density = np.abs(x * y)
normalized_pop_density = (pop_density - np.min(pop_density)) / (np.max(pop_density) - np.min(pop_density))

# Creating a custom linear color map from light green to dark green based on these new normalized densities
colors = [plt.cm.Greens(density) for density in normalized_pop_density]

# Text labels for title, axis labels, etc.
title = "Population Density in Circular City Layout"
xlabel = "X Coordinate"
ylabel = "Y Coordinate"
legendlabel = "Population Density"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size for better visibility
plt.figure(figsize=(8, 8))

# Plot the points with varying sizes and colors
plt.scatter(x, y, c=colors, s=100, edgecolor="black", linewidth=0.5)

# Set the aspect of the plot to be equal
plt.axis("equal")

# Add axis lines
plt.axhline(0, color="gray", linewidth=0.5)
plt.axvline(0, color="gray", linewidth=0.5)

# Add title and axis labels
plt.title(title, fontsize=14)
plt.xlabel(xlabel, fontsize=12)
plt.ylabel(ylabel, fontsize=12)

# Add tick labels
plt.xticks([])
plt.yticks([])

# Add a colorbar for population density
sc = plt.scatter(x, y, c=normalized_pop_density, cmap="Greens")
cb = plt.colorbar(sc)
cb.set_label(legendlabel, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("HR_40.pdf", bbox_inches="tight")
