
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Placeholder data
m_values = np.logspace(1, 2, 10)
costs_small_cases = np.exp(0.05 * m_values)
costs_medium_cases = np.exp(0.1 * m_values + 1)  # Medium cases, higher costs
costs_large_cases = np.exp(0.15 * m_values + 2)  # Large cases, even higher costs
costs_massive_cases = np.exp(0.2 * m_values + 3)  # Massive cases, highest costs

# Create error bands
costs_small_cases_err = costs_small_cases * 0.3
costs_medium_cases_err = costs_medium_cases * 0.4
costs_large_cases_err = costs_large_cases * 0.5
costs_massive_cases_err = costs_massive_cases * 0.6

xlabel = "Number of Cases (m)"
ylabels = [
    r"Average Cost per Case",
    r"Cost Deviation per Case",
]
xlim = [6, 1e2]
ylim = [1e2, 1e6]
title = "Legal Cases Cost Analysis"
suptitle = "Legal Case Complexity and Costs Over Time"
legendlabels = ["Small Cases", "Medium Cases", "Large Cases", "Massive Cases"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axes with specified size (width, height)
fig, (ax1, ax2) = plt.subplots(1, 2, figsize=(12, 6))

# Plot data and curves for the left subplot
ax1.plot(m_values, costs_small_cases, "o", color="darkblue", label=legendlabels[0])
ax1.fill_between(
    m_values,
    costs_small_cases - costs_small_cases_err,
    costs_small_cases + costs_small_cases_err,
    color="darkblue",
    alpha=0.2,
)
# Fit and plot a line
coeffs = np.polyfit(np.log(m_values), np.log(costs_small_cases), 2)
fit_func = np.poly1d(coeffs)
ax1.plot(m_values, np.exp(fit_func(np.log(m_values))), "k--")

ax1.plot(m_values, costs_medium_cases, "s", color="darkgreen", label=legendlabels[1])
ax1.fill_between(
    m_values,
    costs_medium_cases - costs_medium_cases_err,
    costs_medium_cases + costs_medium_cases_err,
    color="darkgreen",
    alpha=0.2,
)
# Fit and plot a line
coeffs = np.polyfit(np.log(m_values), np.log(costs_medium_cases), 2)
fit_func = np.poly1d(coeffs)
ax1.plot(m_values, np.exp(fit_func(np.log(m_values))), "k--")

ax1.set_xscale("log")
ax1.set_yscale("log")
ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabels[0])
ax1.set_title("Average Costs")

# Plot data and curves for the right subplot
ax2.plot(m_values, costs_small_cases, "o", color="darkblue", label=legendlabels[0])
ax2.fill_between(
    m_values,
    costs_small_cases - costs_small_cases_err,
    costs_small_cases + costs_small_cases_err,
    color="darkblue",
    alpha=0.2,
)
# Fit and plot a line
coeffs = np.polyfit(np.log(m_values), np.log(costs_small_cases), 2)
fit_func = np.poly1d(coeffs)
ax2.plot(m_values, np.exp(fit_func(np.log(m_values))), "k--")

ax2.plot(m_values, costs_medium_cases, "s", color="darkgreen", label=legendlabels[1])
ax2.fill_between(
    m_values,
    costs_medium_cases - costs_medium_cases_err,
    costs_medium_cases + costs_medium_cases_err,
    color="darkgreen",
    alpha=0.2,
)
# Fit and plot a line
coeffs = np.polyfit(np.log(m_values), np.log(costs_medium_cases), 2)
fit_func = np.poly1d(coeffs)
ax2.plot(m_values, np.exp(fit_func(np.log(m_values))), "k--")

ax2.plot(m_values, costs_large_cases, "d", color="darkred", label=legendlabels[2])
ax2.fill_between(
    m_values,
    costs_large_cases - costs_large_cases_err,
    costs_large_cases + costs_large_cases_err,
    color="darkred",
    alpha=0.2,
)
# Fit and plot a line
coeffs = np.polyfit(np.log(m_values), np.log(costs_large_cases), 2)
fit_func = np.poly1d(coeffs)
ax2.plot(m_values, np.exp(fit_func(np.log(m_values))), "k--")

ax2.plot(m_values, costs_massive_cases, "^", color="darkorange", label=legendlabels[3])
ax2.fill_between(
    m_values,
    costs_massive_cases - costs_massive_cases_err,
    costs_massive_cases + costs_massive_cases_err,
    color="darkorange",
    alpha=0.2,
)
# Fit and plot a line
coeffs = np.polyfit(np.log(m_values), np.log(costs_massive_cases), 2)
fit_func = np.poly1d(coeffs)
ax2.plot(m_values, np.exp(fit_func(np.log(m_values))), "k--")

ax2.set_xlim(xlim)
ax2.set_ylim(ylim)
ax2.set_xscale("log")
ax2.set_yscale("log")
ax2.set_xlabel(xlabel)
ax2.set_ylabel(ylabels[1])
ax2.set_title("Cost Deviations")
ax2.legend(frameon=True)

fig.suptitle(suptitle, fontsize=14)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save plot
plt.tight_layout(rect=[0, 0, 1, 0.96])
plt.savefig("HR_69.pdf", bbox_inches="tight")
