
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.collections import PatchCollection
from matplotlib.patches import Rectangle

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Number of data points
n = 6

# Generate new data for average monthly temperatures (in °C) over six months
months = np.arange(1, n+1, 1)
avg_temp = np.array([22.5, 24.0, 26.5, 28.0, 30.0, 31.5])  # Simulated average temperatures

# Dummy errors (above and below) representing temperature uncertainty
xerr = np.random.rand(2, n) * 0.2 + 0.1  # Small variation in measurement timing
yerr = np.random.rand(2, n) * 2.0 + 0.5  # Daily temperature fluctuations

# Chart Labels and Title
title = "Average Monthly Temperature with Uncertainty"
xlabel = "Month"
ylabel = "Temperature (°C)"

def make_error_boxes(
    ax, xdata, ydata, xerror, yerror, facecolor, edgecolor="none", alpha=0.5
):
    # Loop over data points; create box from errors at each point
    errorboxes = [
        Rectangle((x - xe[0], y - ye[0]), xe.sum(), ye.sum())
        for x, y, xe, ye in zip(xdata, ydata, xerror.T, yerror.T)
    ]

    # Create patch collection with specified colour/alpha
    pc = PatchCollection(
        errorboxes, facecolor=facecolor, alpha=alpha, edgecolor=edgecolor
    )

    # Add collection to axes
    ax.add_collection(pc)

    # Plot errorbars
    artists = ax.errorbar(
        xdata, ydata, xerr=xerror, yerr=yerror, fmt="none", ecolor="k"
    )

    return artists

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axes
fig, ax = plt.subplots(figsize=(8, 6))
color = "#2a9d8f"  # Nice greenish color for environmental theme

# Call function to create error boxes
_ = make_error_boxes(ax, months, avg_temp, xerr, yerr, facecolor=color)

# Set titles and labels
ax.set_title(title, fontsize=16, fontweight='bold', color="#264653")
ax.set_xlabel(xlabel, fontsize=14, fontweight='bold', color="#264653")
ax.set_ylabel(ylabel, fontsize=14, fontweight='bold', color="#264653")

# Customize ticks and labels
ax.tick_params(axis='both', which='major', labelsize=12)
ax.set_xticks(months)
ax.set_xticklabels(['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'])

# Enhance grid visibility
ax.grid(True, linestyle='--', alpha=0.7)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("HR_78.pdf", bbox_inches="tight")
