
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.collections import PatchCollection
from matplotlib.patches import Rectangle

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)

# Number of data points
n = 6

# Dummy data for demonstration
x = np.arange(0, n, 1)
y = np.random.rand(n) * 10.0  # Simulated values for crop yield

# Dummy errors (above and below) representing measurement uncertainty
xerr = np.random.rand(2, n) + 0.3  # Variation due to soil quality
yerr = np.random.rand(2, n) + 0.5  # Variation due to weather conditions

# Labels and titles
title = "Uncertainty in Crop Yield Measurements"
xlabel = "Region Number"
ylabel = "Crop Yield (Tonnes/Hectare)"

def make_error_boxes(
    ax, xdata, ydata, xerror, yerror, facecolor, edgecolor="none", alpha=0.5
):
    # Loop over data points; create box from errors at each point
    errorboxes = [
        Rectangle((x - xe[0], y - ye[0]), xe.sum(), ye.sum())
        for x, y, xe, ye in zip(xdata, ydata, xerror.T, yerror.T)
    ]

    # Create patch collection with specified colour/alpha
    pc = PatchCollection(
        errorboxes, facecolor=facecolor, alpha=alpha, edgecolor=edgecolor
    )

    # Add collection to axes
    ax.add_collection(pc)

    # Plot errorbars
    artists = ax.errorbar(
        xdata, ydata, xerr=xerror, yerr=yerror, fmt="none", ecolor="brown"
    )

    return artists


# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axes
fig, ax = plt.subplots(figsize=(7, 5))
color = "#2c9c3e"  # Green color to represent agriculture

# Call function to create error boxes
_ = make_error_boxes(ax, x, y, xerr, yerr, facecolor=color)

# Set titles and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("HR_79.pdf", bbox_inches="tight")
