
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

number_of_bins = 40

# Example health metrics data sets to compare
number_of_data_points = 387
labels = ["Group A - Blood Pressure", "Group B - Cholesterol", "Group C - Blood Sugar"]
data_sets = [
    np.random.normal(120, 15, number_of_data_points),  # Blood Pressure
    np.random.normal(200, 40, number_of_data_points), # Cholesterol
    np.random.normal(90, 20, number_of_data_points),  # Blood Sugar
]

# The bin_edges are the same for all of the histograms
hist_range = (np.min(data_sets), np.max(data_sets))
bin_edges = np.linspace(hist_range[0], hist_range[1], number_of_bins + 1)
centers = bin_edges[:-1] + np.diff(bin_edges) / 2

# Labels and title for the health data
xlabel = "Number of Observations"
ylabel = "Measurement Range"
title = "Health Metrics Distribution Across Different Groups"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Cycle through and plot each histogram
# Computed quantities to aid plotting
colors = ["#4caf50", "#2196f3", "#ff9800"]
binned_data_sets = [
    np.histogram(d, range=hist_range, bins=number_of_bins)[0] for d in data_sets
]
binned_maximums = np.max(binned_data_sets, axis=1)
x_locations = np.arange(0, sum(binned_maximums), np.max(binned_maximums))

fig, ax = plt.subplots(figsize=(8, 6))
for x_loc, binned_data, color in zip(x_locations, binned_data_sets, colors):
    lefts = x_loc - 0.5 * binned_data
    ax.barh(
        centers, binned_data, height=np.diff(bin_edges), left=lefts, color=color, edgecolor='black'
    )

# Set the x-axis labels
ax.set_xticks(x_locations)
ax.set_xticklabels(labels)

# Apply custom styling to labels and title
ax.set_xlabel(xlabel, fontsize=12, fontweight='bold')
ax.set_ylabel(ylabel, fontsize=12, fontweight='bold')
ax.set_title(title, fontsize=14, fontweight='bold')

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("HR_90.pdf", bbox_inches="tight")
