
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.ticker import FuncFormatter

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Define the increments and decrements for each step of the waterfall chart
increments = [50, -20, 70, -30, 25, -15, 40, -10, 60, -5, 35]

# Define the labels for each step
labels = [
    "Initial",
    "Tree Growth",
    "Air Quality",
    "Deforestation",
    "Water Conservation",
    "Urbanization",
    "Reforestation",
    "Pollution",
    "Biodiversity",
    "Soil Health",
    "Final",
]

# Determine starting point and end point
start_value = 400
end_value = start_value + sum(increments)

# Calculate the bottom of each bar (cumulative)
bottoms = np.hstack(([start_value], np.cumsum(increments)[:-1])) + start_value

# Axes Limits and Labels
ylabel_value = "Environmental Index"
title = "Forest Ecosystem Health Over Time"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the colors based on increment or decrement
colors = [
    "#76c7c0" if x > 0 else "#d95f02" for x in increments
]

fig, ax = plt.subplots(figsize=(12, 7))

# Plot bars
bars = ax.bar(labels, increments, bottom=bottoms, color=colors)

# Plot lines connecting the tops of each bar
for i in range(len(increments) - 1):  # Exclude the last increment
    start_top = bottoms[i] + increments[i]
    end_top = bottoms[i + 1] + increments[i + 1]
    ax.plot([i, i + 1], [start_top, end_top], color="k", linestyle="--", linewidth=1)

# Annotate bars with value labels
for i, bar in enumerate(bars):
    height = bar.get_height()
    ax.annotate(
        f"{height}",
        xy=(bar.get_x() + bar.get_width() / 2, bar.get_y() + height),
        xytext=(
            0,
            3 if height > 0 else -12,
        ),  # 3 points vertical offset or -12 if negative
        textcoords="offset points",
        ha="center",
        va="bottom",
    )

# Set the y-axis label and title
ax.set_ylabel(ylabel_value)
ax.set_title(title)

# Format y-axis with a more readable format
formatter = FuncFormatter(lambda y, _: f"{y:.0f}")
ax.yaxis.set_major_formatter(formatter)

# Remove x-axis line and ticks
ax.spines["bottom"].set_visible(False)
ax.xaxis.set_ticks([])

# Set grid
ax.grid(True, axis="y", linestyle="--", linewidth=0.7, alpha=0.7)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("HR_92.pdf", bbox_inches="tight")
