
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.tri as tri

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Create hypothetical data for religion domain
n_activities = 48
n_communities = 12
min_participation_rate = 0.1
participation_rates = np.linspace(min_participation_rate, 1.0, n_communities)

angles = np.linspace(0, 2 * np.pi, n_activities, endpoint=False)
angles = np.repeat(angles[..., np.newaxis], n_communities, axis=1)
angles[:, 1::2] += np.pi / n_activities

x = (participation_rates * np.cos(angles)).flatten()
y = (participation_rates * np.sin(angles)).flatten()

# Hypothetical participation scores for religious activities
z = (np.sin(participation_rates * np.pi) * np.cos(2 * angles)).flatten()

# Labels and titles
xlabel = "Participation Rate (x)"
ylabel = "Participation Rate (y)"
title = "Participation Rates in Religious Activities Across Communities"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the Triangulation; no triangles so Delaunay triangulation created.
triang = tri.Triangulation(x, y)

# Mask off unwanted triangles.
triang.set_mask(np.hypot(x[triang.triangles].mean(axis=1), y[triang.triangles].mean(axis=1)) < min_participation_rate)

fig1, ax1 = plt.subplots(figsize=(10, 8))
ax1.set_aspect("equal")
tpc = ax1.tripcolor(triang, z, shading="flat", cmap="viridis")
cbar = fig1.colorbar(tpc)
cbar.set_label('Participation Scores')

ax1.set_title(title)
ax1.set_xlabel(xlabel)
ax1.set_ylabel(ylabel)

# Additional styling
ax1.tick_params(axis='both', which='major', labelsize=10)
ax1.grid(True, linestyle='--', alpha=0.6)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("HR_95.pdf", bbox_inches="tight")
