# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Generate some political data
liberal_support = [20, 25, 27, 30, 35, 40, 45, 50, 55, 57, 60]  # Liberal Support (in %)
conservative_support = [
    30,
    35,
    37,
    38,
    40,
    43,
    45,
    47,
    50,
    52,
    55,
]  # Conservative Support (in %)
years = list(range(2010, 2021))
labels = ["Liberal Support", "Conservative Support"]
xmainlabel = "Years"
xmainlim = [2009, 2021]
xmainticks = list(range(2010, 2021))
ymainlabel = "Support Percentage (%)"
ymainlim = [0, 100]
ymainticks = list(range(0, 101, 10))

xinsetlim = [2015, 2021]
xinsetticks = list(range(2016, 2022))
yinsetlim = [0, 60]
yinsetticks = list(range(0, 61, 10))

title_main = "Political Support Trends Over Time"
title_inset = "Zoomed View (2015-2021)"

# Coordinates for lines connecting the plots (main and inset)
mainplotline = [(2015.5, 0), (2021, 0)]
maininsetline = [(2015, 0), (2021, 0)]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create main plot with adjusted bar widths and white borders
fig, ax_main = plt.subplots(figsize=(10, 6))
bar_width = 0.4  # Slightly less than the year interval to create a gap

# Liberal support bar
ax_main.bar(
    [year - bar_width / 2 for year in years],
    liberal_support,
    width=bar_width,
    color="#FF5733",  # Red for Liberal Support
    align="center",
    label=labels[0],
    edgecolor="white",
)

# Conservative support bar
ax_main.bar(
    [year + bar_width / 2 for year in years],
    conservative_support,
    width=bar_width,
    color="#337AB7",  # Blue for Conservative Support
    align="center",
    label=labels[1],
    edgecolor="white",
)

ax_main.set_xlabel(xmainlabel)
ax_main.set_xlim(xmainlim)
ax_main.set_xticks(xmainticks)
ax_main.set_ylabel(ymainlabel)
ax_main.set_ylim(ymainlim)
ax_main.set_yticks(ymainticks)
ax_main.legend(loc="upper left", prop={"size": 12})
ax_main.set_title(title_main)

# Inset plot configuration
ax_inset = fig.add_axes([0.55, 0.58, 0.3, 0.3])
ax_inset.bar(
    [year - bar_width / 2 for year in years[5:]],
    liberal_support[5:],
    width=bar_width,
    color="#FF5733",
    align="center",
    edgecolor="white",
)
ax_inset.bar(
    [year + bar_width / 2 for year in years[5:]],
    conservative_support[5:],
    width=bar_width,
    color="#337AB7",
    align="center",
    edgecolor="white",
)
ax_inset.set_xlim(xinsetlim)  # Zoom in on the years 2015-2021
ax_inset.set_xticks(xinsetticks)
ax_inset.set_ylim(yinsetlim)
ax_inset.set_yticks(yinsetticks)
ax_inset.set_title(title_inset)

# Adding lines to connect the plots.
# Coordinates of the main plot corners
main_plot_left = ax_main.transData.transform_point(mainplotline[0])
main_plot_right = ax_main.transData.transform_point(mainplotline[1])

# Coordinates of the inset corners
inset_left = ax_inset.transData.transform_point(maininsetline[0])
inset_right = ax_inset.transData.transform_point(maininsetline[1])

# Transform to figure coordinates for annotation
main_plot_left = fig.transFigure.inverted().transform(main_plot_left)
main_plot_right = fig.transFigure.inverted().transform(main_plot_right)
inset_left = fig.transFigure.inverted().transform(inset_left)
inset_right = fig.transFigure.inverted().transform(inset_right)

# Draw lines connecting corners
fig.add_artist(
    plt.Line2D(
        (main_plot_left[0], inset_left[0]),
        (main_plot_left[1], inset_left[1]),
        color="gray",
    )
)
fig.add_artist(
    plt.Line2D(
        (main_plot_right[0], inset_right[0]),
        (main_plot_right[1], inset_right[1]),
        color="gray",
    )
)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("PIP_66.pdf", bbox_inches="tight")
