# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
appliance_energy = {
    "Air Conditioner": np.random.normal(loc=(300, 500), scale=50, size=(50, 2)),
    "Refrigerator": np.random.normal(loc=(200, 400), scale=40, size=(30, 2)),
    "Washing Machine": np.random.normal(loc=(400, 600), scale=60, size=(40, 2)),
    "Heater": np.random.normal(loc=(500, 700), scale=70, size=(60, 2)),
    "Dishwasher": np.random.normal(loc=(250, 450), scale=45, size=(70, 2)),
    "Microwave": np.random.normal(loc=(100, 200), scale=30, size=(45, 2)),
}

title = "Energy Consumption and User Satisfaction of Various Appliances"
xlabel = "Energy Consumption (kWh)"
ylabel = "User Satisfaction Score"
# Inset plot configuration
insetaxes = [0.3, 0.6, 0.3, 0.3]
insetxlim = [100, 500]  # Focus on several appliances with similar characteristics
insetylim = [200, 600]
insetxticks = [150, 250, 350, 450]
insetyticks = [200, 300, 400, 500]

# Arrow and annotation configuration
arrowstart = (400, 500)  # Arrow start for focused area
arrowend = (0.30, 0.7)  # Relative inset arrow end
annotaterecx = [200, 400]  # Annotation range for X
annotaterecy = [300, 500]  # Annotation range for Y

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Colors for appliances
colors = {
    "Air Conditioner": "#ff9999",
    "Refrigerator": "#66b3ff",
    "Washing Machine": "#99ff99",
    "Heater": "#ffcc99",
    "Dishwasher": "#c2c2f0",
    "Microwave": "#ffb3e6",
}

# Markers for appliances
markers = {
    "Air Conditioner": "o",
    "Refrigerator": "s",
    "Washing Machine": "^",
    "Heater": "D",
    "Dishwasher": "P",
    "Microwave": "X",
}

# Create the scatter plot
fig, ax = plt.subplots(figsize=(10, 8))
for appliance, data in appliance_energy.items():
    ax.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[appliance],
        label=appliance,
        alpha=0.6,
        edgecolors="w",
        s=100,
        marker=markers[appliance],
    )

# Enclosing rectangle for annotation
rect = plt.Rectangle(
    (annotaterecx[0], annotaterecy[0]),
    annotaterecx[1] - annotaterecx[0],
    annotaterecy[1] - annotaterecy[0],
    linewidth=1,
    edgecolor="black",
    facecolor="none",
)
ax.add_patch(rect)

# Create the inset with zoomed-in view
ax_inset = fig.add_axes(insetaxes)
for appliance, data in appliance_energy.items():
    ax_inset.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[appliance],
        alpha=0.6,
        edgecolors="w",
        s=40,
        marker=markers[appliance],
    )
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")  # Add black border to the inset
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=1),
)

# Adding titles and labels
ax.set_title(title)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.legend()

# ===================
# Part 4: Saving Output
# ===================
# Show the plot
plt.tight_layout()
plt.savefig("PIP_76.pdf", bbox_inches="tight")
